<?php

namespace AmgGroup\Tests;

use AmgGroup\Smarty;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

// Make sure our mocks are loaded
require_once __DIR__ . '/Mock/SmartyMock.php';
require_once __DIR__ . '/Mock/ConfigMock.php';

class SmartyTest extends TestCase
{
    /**
     * Test that the default template directory is set correctly
     */
    public function testDefaultTemplateDir()
    {
        $smarty = Smarty::getInstance();

        // Get the template directory
        $templateDir = $smarty->getTemplateDir();

        // The template directory should be an array with at least one element
        $this->assertIsArray($templateDir);
        $this->assertNotEmpty($templateDir);

        // The first element should be the default template directory
        $expectedPath = realpath(__DIR__ . '/../src') . '/../templates/';
        $this->assertEquals(realpath($expectedPath), realpath($templateDir[0]));
    }

    /**
     * Test that the default compile directory is set correctly
     */
    public function testDefaultCompileDir()
    {
        $smarty = Smarty::getInstance();

        // Get the compile directory
        $compileDir = $smarty->getCompileDir();

        // The expected compile directory
        $expectedPath = realpath(__DIR__ . '/../src') . '/../templates_c/';
        $this->assertEquals(realpath($expectedPath), realpath($compileDir));
    }

    /**
     * Test that the default cache directory is set correctly
     */
    public function testDefaultCacheDir()
    {
        $smarty = Smarty::getInstance();

        // Get the cache directory
        $cacheDir = $smarty->getCacheDir();

        // The expected cache directory
        $expectedPath = realpath(__DIR__ . '/../src') . '/../cache/';
        $this->assertEquals(realpath($expectedPath), realpath($cacheDir));
    }

    /**
     * Test that the default config directory is set correctly
     */
    public function testDefaultConfigDir()
    {
        $smarty = Smarty::getInstance();

        // Get the config directory
        $configDir = $smarty->getConfigDir();

        // The config directory should be an array with at least one element
        $this->assertIsArray($configDir);
        $this->assertNotEmpty($configDir);

        // The first element should be the default config directory
        $expectedPath = realpath(__DIR__ . '/../src') . '/../config/';
        $this->assertEquals(realpath($expectedPath), realpath($configDir[0]));
    }

    /**
     * Test that the singleton pattern is working correctly
     */
    public function testSingletonPattern()
    {
        $instance1 = Smarty::getInstance();
        $instance2 = Smarty::getInstance();

        // Both instances should be the same object
        $this->assertSame($instance1, $instance2);
    }

    /**
     * Test that the constructor sets the paths correctly
     */
    public function testConstructorSetsPaths()
    {
        // We need to use reflection to test the private constructor
        $reflection = new ReflectionClass(Smarty::class);
        $constructor = $reflection->getMethod('__construct');
        $constructor->setAccessible(true);

        // Create a mock of the Smarty class
        $smartyMock = $this->getMockBuilder(Smarty::class)
            ->disableOriginalConstructor()
            ->getMock();

        // We expect these methods to be called exactly once
        $smartyMock->expects($this->once())
            ->method('setTemplateDir');

        $smartyMock->expects($this->once())
            ->method('setCompileDir');

        $smartyMock->expects($this->once())
            ->method('setCacheDir');

        $smartyMock->expects($this->once())
            ->method('setConfigDir');

        // Call the constructor
        $constructor->invoke($smartyMock);
    }
}
