<?php

namespace AmgGroup;

use AmgGroup\Config;
use Smarty\Template;

/**
 * Class Smarty
 *
 * Extends the core Smarty template engine to pre-configure directories and defaults.
 * Refactored as a Singleton to ensure a single instance is used throughout the application.
 *
 * @TODO add a check for the required Javascript and CSS files to allow various sections to work!
 */
class Smarty extends \Smarty\Smarty
{
    /**
     * Holds the single instance of Smarty.
     *
     * @var Smarty|null
     */
    private static ?Smarty $instance = null;

    private string $lastTemplate;

    private Config $local_config;

    private \AmgGroup\Logger $logger;

    private string $JsBase = '../public_html/js/';

    private string $CssBase = '../public_html/css/';

    /**
     * Private constructor.
     * Sets default directories and development-friendly configuration.
     */
    private function __construct()
    {
        parent::__construct();

        // Configure any global site data
        $this->local_config = Config::getInstance();
        $reflection = new \ReflectionClass(\Composer\Autoload\ClassLoader::class);
        $projectRoot = dirname($reflection->getFileName().'/../../');

        // Set template directories from config if available, otherwise use defaults
        $templateDir = $this->local_config->get('smarty.template_dir') ?? $projectRoot . '/templates/';
        $compileDir = $this->local_config->get('smarty.compile_dir') ?? $projectRoot . '/templates_c/';
        $cacheDir = $this->local_config->get('smarty.cache_dir') ?? $projectRoot . '/cache/';
        $configDir = $this->local_config->get('smarty.config_dir') ?? $projectRoot . '/config/';

        $this->logger->info("Template Dir : $templateDir");
        $this->logger->info("Compile Dir : $compileDir");
        $this->logger->info("Cache Dir : $cacheDir");
        $this->logger->info("Config Dir : $configDir");
        $this->setTemplateDir($templateDir);
        $this->setCompileDir($compileDir);
        $this->setCacheDir($cacheDir);
        $this->setConfigDir($configDir);
        $this->assign('base_url', '');
        // Register custom functions
        $this->registerPlugin('function', 'help_feedback_icons', [$this, 'smartyHelpFeedbackIcons']);

        // Configure any global site data
        $this->assign('site', $this->local_config->get('project'));
        $this->assign('menu', $this->local_config->get('menu'));

        // Development settings
        // Set debug based on the config file
        $this->debugging = $this->local_config->get('debug.enable') === true;
        // If we have debug enabled, then caching will be false, otherwise use cache
        $this->caching = !$this->debugging;
        $this->compile_check = true;


        /** @var TYPE_NAME $this */
        $this->assign('cssFiles', $this->local_config->get('css'));;
        /** @var TYPE_NAME $this */
        $this->assign('javascriptFiles', $this->local_config->get('javascript'));

        // Register custom modifiers for debugging
        /** @var TYPE_NAME $this */
        $this->registerPlugin('modifier', 'print_r', 'print_r');
        /** @var TYPE_NAME $this */
        $this->registerPlugin('modifier', 'var_dump', 'var_dump');
        /** @var TYPE_NAME $this */
        $this->registerPlugin('modifier', 'json_encode', 'json_encode');
    }

    /**
     * If $template is null/""/false, then display the last template we used
     *
     * @param $template
     * @param $cache_id
     * @param $compile_id
     * @return void
     * @throws \Smarty\Exception
     */
    public function display($template = null, $cache_id = null, $compile_id = null): void
    {
        if (empty($template)) {
            $template = $this->lastTemplate;
        } else {
            $this->lastTemplate = $template;
        }
        // Ensure we have the $error variable set to something, makes template logic easier
        if (!$this->getTemplateVars('error')) {
            /** @var TYPE_NAME $this */
            $this->assign('error', null);
        }
        parent::display($template, $cache_id, $compile_id);
    }

    /**
     * Provides access to the single instance of Smarty.
     *
     * @return Smarty
     */
    public static function getInstance(): Smarty
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Prevent cloning of the instance.
     */
    private function __clone()
    {
    }

    /**
     * Prevent unserialization of the instance.
     */
    public function __wakeup()
    {
    }

    /*
     * Smarty Plugins/Functions etc.
     */

    /**
     * Smarty plugin to add help and feedback icons to the page
     *
     * @param array $params Parameters passed from the Smarty template
     * @param Template $template The Smarty template instance
     * @return string HTML output for help and feedback icons
     */
    public function smartyHelpFeedbackIcons(array $params, Template $template): string
    {
        // Default parameters
        $defaults = [
            'help_title' => 'Help',
            'help_content' => 'No help content has been provided for this page.',
            'feedback_title' => 'Submit Feedback',
            'feedback_url' => '/submitFeedback',
            'icon_position' => 'middle-right', // Options: top-right, middle-right, bottom-right
            'help_icon' => 'fa-circle-question',
            'feedback_icon' => 'fa-comment-dots'
        ];

        // Merge passed parameters with defaults
        $params = array_merge($defaults, $params);

        // Position styling based on icon_position
        $position_styles = [
            'top-right' => 'top: 100px; right: 20px;',
            'middle-right' => 'top: 50%; transform: translateY(-50%); right: 20px;',
            'bottom-right' => 'bottom: 100px; right: 20px;'
        ];

        // Get the position style
        $position_style = $position_styles[$params['icon_position']] ?? $position_styles['middle-right'];

        // Capture all Smarty variables for diagnostic information
        $allVars = $template->getTemplateVars();

        // Remove potentially sensitive data or large objects
        if (isset($allVars['password'])) unset($allVars['password']);
        if (isset($allVars['current_user'])) unset($allVars['current_user']);

        // Get the current template name
        $reflection = new \ReflectionProperty(get_class($template), 'source');
        $reflection->setAccessible(true);
        $source = $reflection->getValue($template);
        $template_info = $source->name ?? 'Unknown template';


        // Get the current URL
        $current_url = htmlspecialchars($_SERVER['REQUEST_URI'] ?? '', ENT_QUOTES, 'UTF-8');

        // Encode template data for form submission
        $template_data = htmlspecialchars(json_encode([
            'template' => $template_info,
            'variables' => $allVars
        ]), ENT_QUOTES, 'UTF-8');

        // Generate HTML for the icons and modals
        $output = <<<HTML
        <!-- Help & Feedback Icons -->
        <div class="help-feedback-icons" style="position: fixed; {$position_style} z-index: 1050;">
            <div class="d-flex flex-column gap-2">
                <button type="button" class="btn btn-primary rounded-circle p-2 shadow" data-bs-toggle="modal" data-bs-target="#helpModal" title="{$params['help_title']}">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
    <circle cx="12" cy="12" r="10"></circle>
    <path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"></path>
    <line x1="12" y1="17" x2="12.01" y2="17"></line>
  </svg>
<!-- <i class="fas {$params['help_icon']} fs-5"></i> -->
                </button>
                <button type="button" class="btn btn-info rounded-circle p-2 shadow" data-bs-toggle="modal" data-bs-target="#feedbackModal" title="{$params['feedback_title']}">
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
    <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
  </svg>
<!-- <i class="fas {$params['feedback_icon']} fs-5"></i> -->
                </button>
            </div>
        </div>

        <!-- Help Modal -->
        <div class="modal fade" id="helpModal" tabindex="-1" aria-labelledby="helpModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header bg-primary text-white">
                        <h5 class="modal-title" id="helpModalLabel">{$params['help_title']}</h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        {$params['help_content']}
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Feedback Modal -->
        <div class="modal fade" id="feedbackModal" tabindex="-1" aria-labelledby="feedbackModalLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-info text-white">
                        <h5 class="modal-title" id="feedbackModalLabel">{$params['feedback_title']}</h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <form id="feedbackForm" action="{$params['feedback_url']}" method="POST">
                        <div class="modal-body">
                            <div class="form-group mb-3">
                                <label for="feedbackType">Feedback Type</label>
                                <select class="form-select" id="feedbackType" name="feedbackType" required>
                                    <option value="">Select type...</option>
                                    <option value="bug">Bug Report</option>
                                    <option value="feature">Feature Request</option>
                                    <option value="question">Question</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>
                            <div class="form-group mb-3">
                                <label for="feedbackContent">Your Feedback</label>
                                <textarea class="form-control" id="feedbackContent" name="feedbackContent" rows="5" required placeholder="Please describe your issue or suggestion..."></textarea>
                            </div>
                            <!-- Hidden fields to capture page context -->
                            <input type="hidden" name="pageInfo" value="{$template_data}">
                            <input type="hidden" name="currentUrl" value="{$current_url}">
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" class="btn btn-primary">Submit Ticket</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
HTML;

        return $output;
    }


    private function getFilesInDirectory($directory)
    {
        $files = [];
        $items = scandir($directory);

        foreach ($items as $item) {
            // Skip "." and ".." entries
            if ($item === '.' || $item === '..') {
                continue;
            }

            $fullPath = $directory . '/' . $item;
            if (is_file($fullPath)) {
                $files[] = $item; // Just the filename, no path
            }
        }

        return $files;
    }


}
