<?php

namespace AmgGroup;

/**
 * Session management class
 * 
 * This class provides utilities for PHP session management with special
 * handling for CLI environments.
 * 
 * @package AmgGroup
 */
class Session
{
    /**
     * Get a unique identifier based on the current terminal
     * 
     * Executes the 'tty' command to get the terminal identifier and
     * returns its MD5 hash to be used as a session ID in CLI environments.
     * 
     * @return string MD5 hash of the terminal identifier
     */
    public static function getTtyID(): string
    {
        $tty = exec('tty');
        return md5($tty);
    }

    /**
     * Start a PHP session
     * 
     * Starts a PHP session if one is not already active. In CLI environments,
     * it sets a consistent session ID based on the terminal identifier to
     * maintain session persistence across script executions.
     * 
     * @return void
     */
    public static function start()
    {
        // Only start a session if one is not already set
        if (session_status() !== PHP_SESSION_ACTIVE) {
            // Detect if we are CLI or HTTP
            if (php_sapi_name() === 'cli') {
                // Check if we have a session_id set already
                if (!session_id()) {
                    session_id(self::getTtyID());
                }
            }
            session_start();
            return;
        } else {
            // OK, we already have a session started, so do nothing
            return;
        }
    }
}
