# AMG Router

A basic routing class for PHP applications.

## Features

- Simple mapping of URL paths to handler callbacks.
- Support for dynamic parameters (e.g., `/user/{id}`).
- Support for HTTP methods (GET, POST, etc.).
- Route grouping with shared prefixes and middleware.
- Response objects for better control over HTTP responses.
- Dependency Injection support for controllers via a Container (PSR-11 compatible).
- Dependency Injection for Logger and Config in the Router.
- Custom 404 Not Found handlers.

## Installation

```bash
composer require amg/amg-router
```

## Usage

```php
use AmgGroup\Router;
use AmgGroup\Response;

$router = new Router();

// Register routes
$router->get('/', function() {
    return "Welcome to the homepage!";
});

// Route Groups
$router->group(['prefix' => '/admin', 'middleware' => 'App\Middleware\AuthMiddleware'], function($router) {
    $router->get('/dashboard', function() {
        return "Admin Dashboard";
    });
});

// Nested Groups
$router->group(['prefix' => '/api'], function($router) {
    $router->group(['prefix' => '/v1'], function($router) {
        $router->get('/users', function() {
            return "API Users V1";
        });
    });
});

// Middleware example (Callable)
$authMiddleware = function($args) {
    $next = $args['next'];
    $params = $args['params'];
    
    // Example: Authentication check
    if (!isset($_SESSION['user'])) {
        return new Response("Unauthorized", 401);
    }
    
    // Continue the chain
    return $next($params);
};

// Logging middleware example
$loggingMiddleware = function($args) {
    // Before handling
    error_log("Request started");
    
    $response = $args['next']($args['params']);
    
    // After handling
    error_log("Response status: " . $response->getStatusCode());
    
    return $response;
};

// Apply middleware to a group
$router->group(['prefix' => '/admin', 'middleware' => $authMiddleware], function($router) {
    $router->get('/dashboard', function() {
        return "Admin Dashboard";
    });
});

// Apply middleware to a single route
$router->get('/profile', 'App\Controllers\UserController@index', [$authMiddleware]);

// String-based middleware (Class@method)
$router->get('/api/secure', 'App\Controllers\ApiController@data', ['App\Middleware\AuthMiddleware@handle']);

// String-based handler (Class@method)
$router->get('/api/data', 'App\Controllers\ApiController@data');

// Route Groups with Namespace
$router->group(['prefix' => '/api/v2', 'namespace' => 'App\Controllers\Api'], function($router) {
    // This will resolve to App\Controllers\Api\UserController@index
    $router->get('/users', 'UserController@index');
});

$router->get('/user/{id}', function($params) {
    return "User ID: " . $params['id'];
});

$router->post('/submit', function() {
    return "Form submitted!";
});

// Custom 404 handler
$router->setNotFoundHandler(function() {
    return new \AmgGroup\Response("Custom 404 Page", 404);
});

// Dependency Injection Container Support
// You can pass a PSR-11 compatible container to the router
$router->setContainer($container);
// Or via constructor
// $router = new Router(null, null, $container);

// Dispatch the route and send the response
$response = $router->dispatch($_SERVER['REQUEST_URI'], $_SERVER['REQUEST_METHOD']);
$response->send();
```

## PHP-DI Integration

The router works seamlessly with [PHP-DI](https://php-di.org/). Here's a quick example:

```php
use DI\ContainerBuilder;
use AmgGroup\Router;

$builder = new ContainerBuilder();
$builder->addDefinitions([
    'UserController' => \DI\create('App\Controllers\UserController')
        ->constructor(\DI\get('Database'))
]);
$container = $builder->build();

$router = new Router(null, null, $container);

$router->get('/user/{id}', 'UserController@show');
```

## Testing

```bash
vendor/bin/phpunit
```
