<?php

namespace AmgGroup;

use \Psr\Log\LoggerInterface;
use \Psr\Log\NullLogger;
use \AmgGroup\Logger;
use \AmgGroup\Config;
use \AmgGroup\Response;

/**
 * Class Router
 *
 * A basic routing class to handle mapping URL paths to handler callbacks, including dynamic params.
 *
 * @package Amg
 */
class Router
{
    /**
     * @var array Array mapping URL paths to handler callbacks.
     */
    protected array $routes = [];

    /**
     * @var array Stack of group attributes (prefix, middleware).
     */
    protected array $groupStack = [];

    /**
     * @var LoggerInterface Logger instance for logging.
     */
    protected LoggerInterface $logger;

    /**
     * @var Config Config loader instance for reading configuration options.
     */
    protected Config $config;

    /**
     * Router constructor.
     *
     * @param Config|null $config Config loader instance.
     * @param LoggerInterface|null $logger Logger instance.
     */
    public function __construct(?Config $config = null, ?LoggerInterface $logger = null)
    {
        $this->config = $config ?? Config::getInstance();
        $this->logger = $logger ?? Logger::getInstance();
    }

    /**
     * Register a new route.
     *
     * @param string          $method  HTTP method (GET, POST, etc.)
     * @param string          $route   The URL path for the route (e.g., '/admin').
     * @param string|callable $handler A callback or 'Class@method' string to handle the route.
     * @param array           $middleware Additional middleware for this specific route.
     *
     * @return void
     */
    public function add(string $method, string $route, string|callable $handler, array $middleware = []): void
    {
        // Apply group prefix
        $prefix = '';
        foreach ($this->groupStack as $group) {
            if (isset($group['prefix'])) {
                $prefix .= '/' . trim($group['prefix'], '/');
            }
        }
        $route = $prefix . '/' . trim($route, '/');
        $route = $route === '/' ? '/' : rtrim($route, '/');

        // Collect group middleware
        $allMiddleware = [];
        foreach ($this->groupStack as $group) {
            if (isset($group['middleware'])) {
                $groupMiddleware = is_array($group['middleware']) ? $group['middleware'] : [$group['middleware']];
                $allMiddleware = array_merge($allMiddleware, $groupMiddleware);
            }
        }

        // Add route-specific middleware
        $allMiddleware = array_merge($allMiddleware, $middleware);

        // Replace dynamic parameters (e.g., {id}) with regex patterns
        $pattern = preg_replace('/\{([a-zA-Z0-9_]+)\}/', '(?P<\1>[a-zA-Z0-9_-]+)', $route);
        $this->routes[strtoupper($method)][$pattern] = [
            'handler' => $handler,
            'middleware' => $allMiddleware,
        ];

        //$this->logger->info("Route added", ['method' => $method, 'route' => $route, 'pattern' => $pattern]);
    }

    /**
     * Create a route group with shared attributes.
     *
     * @param array    $attributes Group attributes (prefix, middleware)
     * @param callable $callback   Callback to register routes within the group
     *
     * @return void
     */
    public function group(array $attributes, callable $callback): void
    {
        $this->groupStack[] = $attributes;
        call_user_func($callback, $this);
        array_pop($this->groupStack);
    }

    /**
     * Shortcut for GET route
     */
    public function get(string $route, string|callable $handler, array $middleware = []): void
    {
        $this->add('GET', $route, $handler, $middleware);
    }

    /**
     * Shortcut for POST route
     */
    public function post(string $route, string|callable $handler, array $middleware = []): void
    {
        $this->add('POST', $route, $handler, $middleware);
    }

    /**
     * @var string|callable|null Custom 404 handler
     */
    protected $notFoundHandler = null;

    /**
     * Dispatch the appropriate route based on the current request URI and method.
     *
     * @param string $uri    The current request URI.
     * @param string $method The HTTP request method.
     *
     * @return Response
     */
    public function dispatch(string $uri, string $method = 'GET'): Response
    {
        // Parse the URL to remove any query string.
        $parsedUrl = parse_url($uri, PHP_URL_PATH);
        // Normalize the URI by removing trailing slash, but keep root as /
        $normalizedUri = $parsedUrl === '/' ? '/' : rtrim($parsedUrl, '/');
        $method = strtoupper($method);

        //$this->logger->info("Attempting to dispatch route : Method=>$method, URI=>$normalizedUri");

        if (isset($this->routes[$method])) {
            foreach ($this->routes[$method] as $routePattern => $routeData) {
                // Match the URI against the route pattern
                if (preg_match('#^' . $routePattern . '$#', $normalizedUri, $matches)) {
                    // Extract named params from matches
                    $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);

                    //$this->logger->info("Route matched : route=>$routePattern, params=>", $params);

                    $handler = $routeData['handler'];
                    $middleware = $routeData['middleware'];

                    // Execute middleware chain
                    return $this->runMiddleware($middleware, $handler, $params);
                }
            }
        }

        $this->logger->warning("Route not found, dispatching 404 : Method=>$method, URI=>$normalizedUri");
        return $this->handleNotFound();
    }

    /**
     * Set a custom 404 handler.
     */
    public function setNotFoundHandler(string|callable $handler): void
    {
        $this->notFoundHandler = $handler;
    }

    /**
     * Handle 404 Not Found.
     *
     * @return Response
     */
    protected function handleNotFound(): Response
    {
        if ($this->notFoundHandler) {
            $result = $this->callHandler($this->notFoundHandler);
            if ($result instanceof Response) {
                return $result;
            }
            return new Response((string) $result, 404);
        }

        return new Response("404 Not Found", 404);
    }

    /**
     * Run the middleware chain.
     *
     * @param array           $middleware
     * @param string|callable $handler
     * @param array           $params
     *
     * @return Response
     */
    protected function runMiddleware(array $middleware, string|callable $handler, array $params = []): Response
    {
        $next = function ($params) use (&$middleware, $handler, &$next) {
            if (empty($middleware)) {
                $result = $this->callHandler($handler, $params);
                if ($result instanceof Response) {
                    return $result;
                }
                return new Response((string) $result);
            }

            $currentMiddleware = array_shift($middleware);
            $result = $this->callHandler($currentMiddleware, ['params' => $params, 'next' => $next]);

            if ($result instanceof Response) {
                return $result;
            }

            return new Response((string) $result);
        };

        return $next($params);
    }

    /**
     * Resolve and call the handler.
     *
     * @param string|callable $handler
     * @param array           $params
     *
     * @return mixed
     */
    protected function callHandler(string|callable $handler, array $params = []): mixed
    {
        if (is_string($handler) && str_contains($handler, '@')) {
            [$class, $method] = explode('@', $handler);
            if (class_exists($class)) {
                $controller = new $class();
                if (method_exists($controller, $method)) {
                    return call_user_func([$controller, $method], $params);
                }
            }
            throw new \RuntimeException("Handler class or method not found: $handler");
        }

        return call_user_func($handler, $params);
    }

    /**
     * Return array of current routes
     *
     * @return array
     */
    public function getRoutes(): array
    {
        return $this->routes;
    }
}
