<?php

use PHPUnit\Framework\TestCase;
use AmgGroup\RegionalData;
use AmgGroup\Country;

class RegionalDataTest extends TestCase
{
    public function testCountryDataLoadsAndIsAccessible(): void
    {
        $countries = new Country();
        $this->assertGreaterThan(0, count($countries));
        $this->assertArrayHasKey('US', $countries->get());
        $this->assertSame('United States', $countries['US']);
    }

    public function testArrayAccessReadOnlyThrows(): void
    {
        $countries = new Country();
        $this->expectException(LogicException::class);
        /** @psalm-suppress InvalidArgument */
        $countries['XX'] = 'Test';
    }

    public function testUnsetThrows(): void
    {
        $countries = new Country();
        $this->expectException(LogicException::class);
        unset($countries['US']);
    }

    public function testIteratorAggregateAndCountable(): void
    {
        $countries = new Country();
        $countViaCountable = count($countries);
        $countViaIter = 0;
        foreach ($countries as $code => $name) {
            $countViaIter++;
            if ($code === 'US') {
                $this->assertSame('United States', $name);
            }
        }
        $this->assertSame($countViaCountable, $countViaIter);
    }

    public function testFindCodeByNameIsCaseAndSpaceInsensitive(): void
    {
        $countries = new Country();
        $this->assertSame('US', $countries->findCodeByName('  united   states  '));
        $this->assertSame('US', $countries->findCodeByName('UNITED STATES'));
    }

    public function testSuggestCodesIncludesExpected(): void
    {
        $countries = new Country();
        $suggestions = $countries->suggestCodes('uni', 10);
        $this->assertIsArray($suggestions);
        $this->assertNotEmpty($suggestions);
        $this->assertContains('US', $suggestions);
        $this->assertLessThanOrEqual(10, count($suggestions));
    }

    public function testClosestCodesContainsExpected(): void
    {
        $countries = new Country();
        // Misspelt France
        $closest = $countries->closestCodes('frnace', 5, 3);
        $this->assertIsArray($closest);
        $this->assertContains('FR', $closest);
    }
}
