# AMG Regional Data (PHP)

Lightweight helper classes and data for common regional datasets (countries, currencies, languages, and locales), focused on `en_GB` names. The project exposes a simple PHP API for loading lists and doing reverse/fuzzy lookups.

Note: The source folders `country-list`, `currency-list`, `language-list`, and `locale-list` are ignored by Git and are only used by the update script; they do not need to be present when using the library.

## Contents
- `RegionalData.php` — main PHP file exporting the helper base `AmgGroup\RegionalData` and typed facades `AmgGroup\Country`, `AmgGroup\Currency`, `AmgGroup\Language`, `AmgGroup\Locale`.
- `data/` — generated data files (multiple formats). The PHP runtime uses the `.php` versions.
- `update` — bash script to clone/update the upstream `*-list` sources and regenerate the `en_GB` data into `data/`.

## Installation

Install via Composer (from your private Composer repo or Packagist if published):

```json
{
  "require": {
    "amggroup/regional-data": "^1.0"
  }
}
```

Or, if using a local path (monorepo):

```json
{
  "repositories": [
    { "type": "path", "url": "./source/php/composer/amg-regionaldata" }
  ],
  "require": {
    "amggroup/regional-data": "*"
  }
}
```

Then autoload as usual via Composer.

## Quick start

```php
<?php
require __DIR__ . '/vendor/autoload.php';

use AmgGroup\src\Country;use AmgGroup\src\Currency;use AmgGroup\src\Language;use AmgGroup\src\Locale;

// Get singleton instances (cached)
$countries = Country::instance();
$currencies = Currency::instance();
$languages = Language::instance();
$locales = Locale::instance();

// Iterate (implements IteratorAggregate)
foreach ($countries as $code => $name) {
    echo "$code => $name\n";
}

// Array-like access (ArrayAccess)
echo $countries['GB']; // e.g. "United Kingdom"

// Countable
echo count($currencies);

// Get raw array
$allLanguages = $languages->get();

// Reverse lookup: find code by (human) name (case/space/diacritics insensitive)
$code = $countries->findCodeByName('United  Kingdom'); // "GB"

// Suggest by substring (normalized)
$suggest = $currencies->suggestCodes('pound', 5); // e.g. ["GBP", ...]

// Fuzzy lookup by Levenshtein (normalized)
$closest = $languages->closestCodes('Portugese', 3, 2); // e.g. ["pt"]
```

### Types and shortcuts

Each dataset has a thin facade over `RegionalData`:

- `AmgGroup\Country` — `Country::instance()` or `new Country()`
- `AmgGroup\Currency` — `Currency::instance()` or `new Currency()`
- `AmgGroup\Language` — `Language::instance()` or `new Language()`
- `AmgGroup\Locale` — `Locale::instance()` or `new Locale()`

All of them expose the same API inherited from `RegionalData`:

- `get(): array` — returns the full map `code => name`.
- `findCodeByName(string $name): ?string` — normalized reverse lookup.
- `suggestCodes(string $query, int $limit = 5): array` — substring suggestions.
- `closestCodes(string $query, int $limit = 3, int $maxDistance = 2): array` — fuzzy suggestions.
- ArrayAccess (`$list[$code]`), `IteratorAggregate`, and `Countable`.

Normalization rules include: lowercasing, trimming, collapsing whitespace, and removing diacritics (when `intl` transliterator is available).

## Data generation (`update` script)

The `update` script is a small bash utility that:
1. Clones/updates the upstream `*-list` repositories into the ignored folders `country-list`, `currency-list`, `language-list`, and `locale-list`.
2. Extracts only `en_GB` data.
3. Writes the results into `data/` in several formats; the runtime uses the `.php` arrays.

Typical usage:

```bash
./update
```

You can run it whenever you need to refresh the data from upstream. Since the source folders are ignored by Git, only the generated `data/` artifacts should change in commits.

## Requirements

- PHP 8.0+ (uses typed properties and union features; adjust as needed)
- Optional: `ext-intl` for better diacritics normalization in reverse/fuzzy lookups

## Versioning & stability

The runtime API is small and stable. Data content depends on upstream sources. Pin versions/commit SHAs in the `update` script for reproducibility if needed.

## License

MIT