<?php

namespace AmgGroup\src;

use ArrayAccess;
use ArrayIterator;
use Countable;
use IteratorAggregate;
use LogicException;

class RegionalData implements ArrayAccess, IteratorAggregate, Countable
{
    private const TYPES = [
        'country' => 'country.php',
        'currency' => 'currency.php',
        'language' => 'language.php',
        'locale' => 'locale.php',
    ];

    // Cache the raw arrays by type so multiple instances don’t re-load
    private static array $dataCache = [];
    // Optional: cache one instance per type
    private static array $instanceCache = [];

    private array $list = [];
    // Lazy-built reverse index: normalized label => code/key
    private ?array $reverse = null;

    public function __construct(string $type)
    {
        if (!array_key_exists($type, self::TYPES)) {
            throw new LogicException("Unknown list type: {$type}");
        }

        if (!isset(self::$dataCache[$type])) {
            $file = __DIR__ . '/../data/' . self::TYPES[$type];
            $data = include $file;
            if (!is_array($data)) {
                throw new LogicException("List file did not return an array: {$file}");
            }
            self::$dataCache[$type] = $data;
        }

        $this->list = self::$dataCache[$type];
    }

    public static function getInstance(string $type): static
    {
        if (!isset(self::$instanceCache[$type])) {
            self::$instanceCache[$type] = new static($type);
        }
        return self::$instanceCache[$type];
    }

    public function get(): array
    {
        return $this->list;
    }

    // Reverse lookup helpers
    private function buildReverse(): void
    {
        if ($this->reverse !== null) {
            return;
        }
        $this->reverse = [];
        foreach ($this->list as $code => $name) {
            $key = self::normalize((string)$name);
            // keep first occurrence if duplicates exist
            if (!isset($this->reverse[$key])) {
                $this->reverse[$key] = (string)$code;
            }
        }
    }

    protected static function normalize(string $s): string
    {
        // lowercase + trim
        $s = trim(mb_strtolower($s));
        // collapse whitespace
        $s = preg_replace('/\s+/u', ' ', $s);
        // strip accents/diacritics when intl is available
        if (class_exists('Transliterator')) {
            $tr = \Transliterator::create('NFD; [:Nonspacing Mark:] Remove; NFC');
            if ($tr) {
                $s = $tr->transliterate($s);
            }
        }
        return $s;
    }

    // Exact/normalized reverse lookup: label -> code
    public function findCodeByName(string $name): ?string
    {
        $this->buildReverse();
        $key = self::normalize($name);
        return $this->reverse[$key] ?? null;
    }

    // Simple substring suggestion over normalized labels
    public function suggestCodes(string $query, int $limit = 5): array
    {
        $queryN = self::normalize($query);
        if ($queryN === '') {
            return [];
        }
        $scores = [];
        foreach ($this->list as $code => $name) {
            $n = self::normalize((string)$name);
            if (str_contains($n, $queryN)) {
                $pos = strpos($n, $queryN);
                $scores[(string)$code] = $pos + (strlen($n) - strlen($queryN));
            }
        }
        asort($scores);
        return array_slice(array_keys($scores), 0, $limit);
    }

    // Fuzzy: return closest codes by Levenshtein distance
    public function closestCodes(string $query, int $limit = 3, int $maxDistance = 2): array
    {
        $queryN = self::normalize($query);
        if ($queryN === '') {
            return [];
        }
        $candidates = [];
        foreach ($this->list as $code => $name) {
            $n = self::normalize((string)$name);
            $d = levenshtein($queryN, $n);
            if ($d <= $maxDistance) {
                $candidates[(string)$code] = $d;
            }
        }
        asort($candidates);
        return array_slice(array_keys($candidates), 0, $limit);
    }

    // ArrayAccess
    public function offsetExists(mixed $offset): bool
    {
        return array_key_exists($offset, $this->list);
    }

    public function offsetGet(mixed $offset): mixed
    {
        return $this->list[$offset] ?? null;
    }

    public function offsetSet(mixed $offset, mixed $value): void
    {
        throw new LogicException('ListData is read-only');
    }

    public function offsetUnset(mixed $offset): void
    {
        throw new LogicException('ListData is read-only');
    }

    // IteratorAggregate
    public function getIterator(): ArrayIterator
    {
        return new ArrayIterator($this->list);
    }

    // Countable
    public function count(): int
    {
        return count($this->list);
    }
}

