<?php

namespace AmgGroup;

use AmgGroup\Logger;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use Psr\Http\Message\ResponseInterface;

/**
 * Class ApiClient
 * ----------------
 * A generic API client with pagination support.
 */
class ApiClient
{
    private array $destructiveMethods=[
        'PATCH',
        'DELETE',
        'CREATE',
        'POST',
        'GET'
    ];
    private Client $client;

    private static ?ApiClient $instance = null;

    private ?string $accessToken = null;

    private Logger $logger;

    private Config $config;
    public ResponseInterface|null $return;
    public int $statusCode;
    private bool $dry = true;

    /**
     * ApiClient constructor.
     *
     * @param array $oauthConfig OAuth2 configuration for the provider.
     */
    private function __construct()
    {
        $this->config = Config::getInstance();
        $this->logger = Logger::getInstance();
        $this->logger->info("ApiClient constructor");

        if (!$this->config->configIsLoaded()) {
            $this->logger->error("Config not loaded");
            throw new \Exception("Config not loaded");
        }
        print_r('Tennant ID : '.$this->config->get('graph.tenant_id'));
        $this->authenticate();

        $this->client = new Client([
            'headers' => [
                'Authorization' => 'Bearer ' . $this->accessToken,
                'Accept' => 'application/json',
            ]
        ]);
    }


    /**
     * Provides access to the single instance of Smarty.
     *
     * @return MSGraph
     */
    public static function getInstance(): ApiClient
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Authenticate and store the access token.
     */
    private function authenticate(): string|null
    {
        $client = new Client();
        try {
            $response = $client->post("https://login.microsoftonline.com/{$this->config->get('graph.tenant_id')}/oauth2/v2.0/token", [
                'form_params' => [
                    'client_id' => $this->config->get('graph.client_id'),
                    'client_secret' => $this->config->get('graph.client_secret'),
                    'grant_type' => 'client_credentials',
                    'scope' => 'https://graph.microsoft.com/.default'
                ]
            ]);

// Check the response body
            $body = json_decode($response->getBody()->getContents(), true); // Decode JSON as array
            $accessToken = $body['access_token']; // Extract the 'access_token'
            return $this->accessToken = $accessToken;
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            echo 'Error: ' . $e->getMessage();
            if ($e->hasResponse()) {
                $body = $e->getResponse()->getBody();
                echo 'Response Body: ' . $body;
            }
            return null;
        }
    }

    /**
     * Perform a GET request with pagination support.
     *
     * @param string $url Base API endpoint.
     * @param array $params Optional query parameters.
     * @param string $nextKey JSON key path to next URL (default: 'next').
     * @return array All combined response data pages.
     */
    public function getAllPages(string $url, array $params = [], string $nextKey = '@odata.nextLink'): array
    {
        $results = [];

        try {
            $fullUrl = $url . '?' . http_build_query($params);
            //echo "Full URL: " . $fullUrl . PHP_EOL;
            while ($fullUrl) {
                $response = $this->client->get($fullUrl);
                $data = json_decode($response->getBody(), true);

                // Append results from the current page
                if (isset($data['value']) && is_array($data['value'])) {
                    $results = array_merge($results, $data['value']);
                }

                // Determine next URL (if pagination is used)
                $fullUrl = $data[$nextKey] ?? null;
            }
        } catch (RequestException $e) {
            echo "Request failed: " . $e->getMessage();
        }

        return $results;
    }

    /**
     * Send a generic HTTP request.
     *
     * @param string $method HTTP method (GET, POST, etc).
     * @param string $url Endpoint URL.
     * @param array $options Guzzle options (headers, json, query, etc).
     * @return ResponseInterface|null
     * @throws GuzzleException
     */
    public function request(string $method, string $url, array $options = []): ?ResponseInterface
    {
        unset($options['json']['id']);
        if ((in_array($method, $this->destructiveMethods)) && $this->dry !== false)  {
            // OK, we are set to Dry Run, so we dont allow any destructive operations
            $this->logger->info->apiRequest('Request to write data :'.$method, ['url'=> $url, 'options'=> $options]);
            $this->statusCode = 200;
        } else {
            if ($method === 'CREATE') {
                /*
                 * We use 'CREATE' as a method name to allow us to separate
                 * types of POST request
                 */
                $method = 'POST';
            }
            try {
                $this->return = $this->client->request($method, $url, $options);
            } catch (RequestException $e) {
                //echo "Request error: " . $e->getMessage();
                $this->logger->error("Request error : ", [$e->getResponse()->getBody()->getContents()]);
                $this->logger->error("Request Type : $method");
                $this->logger->error("Request URL : $url");
                $this->logger->error("Request Options : ", $options);
                $this->return = null;
            }
            $this->statusCode = $this->return ? $this->return->getStatusCode() : 0;
        }
        return $this->return;
    }

}
