### MS Graph Helpers (PHP)

This library provides a small, extensible foundation for working with Microsoft Graph in PHP. It includes ready-to-use clients for users and Conditional Access policies, plus simple OData helpers and a base `MSGraph` class you can extend to target any other Graph endpoint.

- Namespace: `AmgGroup`
- Package: `amggroup/msgraph2`

---

### Installation

```bash
composer require amggroup/msgraph2
```

This package relies on the following AMG components (installed automatically via Composer):

- `amggroup/apiclient` — lightweight HTTP client wrapper (Guzzle under the hood)
- `amggroup/msgraphclientcredentialstokenprovider` — OAuth2 Client Credentials token provider for Microsoft Graph
- `monolog/monolog` (logging), `psr/log`, etc.

---

### Quick Start

#### 1) Create an ApiClient configured for Microsoft Graph

```php
use AmgGroup\ApiClient;
use AmgGroup\MSGraphClientCredentialsTokenProvider; // from amggroup/msgraphclientcredentialstokenprovider

$tenantId     = '00000000-0000-0000-0000-000000000000';
$clientId     = '11111111-1111-1111-1111-111111111111';
$clientSecret = 'your-client-secret';

$tokenProvider = new MSGraphClientCredentialsTokenProvider(
    tenantId: $tenantId,
    clientId: $clientId,
    clientSecret: $clientSecret,
    // Default scope for Graph REST (app permissions)
    scopes: ['https://graph.microsoft.com/.default']
);

$api = new ApiClient([
    'base_uri' => 'https://graph.microsoft.com/v1.0',
    'auth'     => $tokenProvider, // the ApiClient integrates with the provider to inject Bearer tokens
    'headers'  => [
        'Accept'       => 'application/json',
        'Content-Type' => 'application/json',
    ],
]);
```

Notes:
- If you need `beta`, change `base_uri` to `https://graph.microsoft.com/beta`.
- You can add global headers with `setDefaultHeaders()` later.

#### 2) Instantiate a resource client

```php
use AmgGroup\UsersClient;
use AmgGroup\ConditionalAccessClient;

$users = new UsersClient($api);
$ca    = new ConditionalAccessClient($api);

// Optional: enable automatic retry/backoff for 429/5xx
$users->enableBackoff(maxRetries: 5, baseDelayMs: 300);
$ca->enableBackoff(5, 300);
```

---

### OData Query Helpers

Use `GraphQuery` and `Filter` to build `$select`, `$filter`, `$top`, `$orderby`, `$expand`:

```php
use AmgGroup\GraphQuery;
use AmgGroup\Filter;

$q = GraphQuery::create()
    ->select(['id', 'displayName', 'userPrincipalName'])
    ->filter(Filter::startsWith('displayName', 'Alex'))
    ->top(20)
    ->orderBy('displayName');
```

Then pass `$q` into client methods that accept queries.

---

### Users: read and navigate membership

Class: `AmgGroup\UsersClient`

- `findUser(string $ref, ?GraphQuery $q = null): ?array`
    - Finds by UUID or by exact `userPrincipalName`, `mail`, or `displayName`.
    - Returns a single user as array, or `null` if not found or ambiguous.

```php
// Find by UPN / mail / displayName / UUID
$user = $users->findUser('alex@example.com');
if (!$user) {
    // handle not found or ambiguous
}

// With a custom $select
$q = GraphQuery::create()->select('id,displayName,accountEnabled');
$user = $users->findUser('11111111-1111-1111-1111-111111111111', $q);
```

- `getGroups(string $userRef, ?GraphQuery $q = null): array`
    - Lists directory objects the user is a member of (`/users/{id}/memberOf`).

```php
$groups = $users->getGroups('alex@example.com', GraphQuery::create()->select('id,displayName'));
foreach ($groups as $g) {
    // $g['@odata.type'] might be group, directoryRole, etc.
}
```

- `getMember(string $groupRef, string $memberRef): ?array`
    - Convenience for `/groups/{groupId}/members/{memberId}`.
    - Resolves group by `id` or by exact `displayName`/`mailNickname`; resolves user by `id` or via `findUser()`.

```php
$member = $users->getMember('Marketing', 'alex@example.com');
if ($member === null) {
    // not found
}
```

Write operations for users are not provided in `UsersClient` in this package. To write to user resources (create, update, delete), see the "Extend to other endpoints" section below for how to add those in a custom client.

---

### Conditional Access: list, find, create, update, delete

Class: `AmgGroup\ConditionalAccessClient`

- `list(?GraphQuery $q = null, string $indexBy = 'id'): array`
    - Retrieves all policies from `/identity/conditionalAccess/policies` (auto-paginated).
    - Optional `$indexBy` re-indexes the returned array by a field (default `id`). Pass `''` to keep numeric indexes.

```php
$policies = $ca->list(GraphQuery::create()->select('id,displayName,state'));
$single   = $policies['abcd-ef01-...'] ?? null;
```

- `find(string $ref, ?GraphQuery $q = null): ?array`
    - Find by UUID or exact `displayName` (returns `null` if not found or ambiguous).

```php
$policy = $ca->find('Require MFA for admins');
```

- `create(array $policy): ?Psr\Http\Message\ResponseInterface`

```php
$newPolicy = [
    'displayName' => 'Block legacy auth',
    'state'       => 'enabled', // or disabled
    'conditions'  => [/* ... */],
    'grantControls' => [/* ... */],
];

$response = $ca->create($newPolicy);
if ($response && $response->getStatusCode() >= 200 && $response->getStatusCode() < 300) {
    $created = json_decode((string)$response->getBody(), true);
}
```

- `update(string $ref, array $patch): ?ResponseInterface`
    - `ref` can be policy `id` or `displayName`.

```php
$resp = $ca->update('Require MFA for admins', ['state' => 'disabled']);
```

- `delete(string $ref): ?ResponseInterface`

```php
$resp = $ca->delete('abcd1234-abcd-...');
$ok = $resp && $resp->getStatusCode() === 204;
```

---

### Pagination and JSON helpers

All list methods internally auto-paginate through `@odata.nextLink` via the base `ApiClient`.

For single-object GETs, `MSGraph::getJson($path)` decodes JSON and returns `array|null`.

You can access the last raw response via:

```php
$last = $ca->getLastResponse();
$code = $ca->getStatusCode();
```

---

### Timeouts, headers, and retries

- Set or override default headers at runtime:

```php
$users->setDefaultHeaders([
    'ConsistencyLevel' => 'eventual',
]);
```

- Enable retry/backoff for transient errors (429/5xx):

```php
$users->enableBackoff(maxRetries: 5, baseDelayMs: 250);
// $users->disableBackoff();
```

Backoff respects the `Retry-After` header if present, otherwise uses exponential backoff with jitter.

---

### Extending MSGraph for new endpoints

Use the `MSGraph` base class to build your own, highly targeted clients. Typical pattern:

```php
namespace App\Graph;

use AmgGroup\MSGraph;
use AmgGroup\GraphQuery;
use AmgGroup\Filter;
use Psr\Http\Message\ResponseInterface;

final class GroupsClient extends MSGraph
{
    private string $path = '/groups';

    public function list(?GraphQuery $q = null): array
    {
        $params = $q ? $q->toArray() : [];
        return $this->getAllPages($this->path, $params);
    }

    public function find(string $ref, ?GraphQuery $q = null): ?array
    {
        if ($this->isUuid($ref)) {
            return $this->getJson($this->path . '/' . $ref);
        }
        $query = $q ?? GraphQuery::create()->filter(
            Filter::or(
                Filter::eq('displayName', $ref),
                Filter::eq('mailNickname', $ref)
            )
        )->top(2);
        $items = $this->getAllPages($this->path, $query->toArray());
        return count($items) === 1 ? $items[0] : null;
    }

    public function create(array $group): ?ResponseInterface
    {
        return $this->request('POST', $this->path, ['json' => $group]);
    }

    public function update(string $ref, array $patch): ?ResponseInterface
    {
        $id = $this->isUuid($ref) ? $ref : ($this->find($ref)['id'] ?? null);
        if ($id === null) return null;
        return $this->request('PATCH', $this->path . '/' . $id, ['json' => $patch]);
    }

    public function delete(string $ref): ?ResponseInterface
    {
        $id = $this->isUuid($ref) ? $ref : ($this->find($ref)['id'] ?? null);
        if ($id === null) return null;
        return $this->request('DELETE', $this->path . '/' . $id);
    }
}
```

Key building blocks from `MSGraph` you can use:

- `request($method, $path, $options)` — wraps `ApiClient->request` with optional retry/backoff
- `getAllPages($path, $query)` — returns a merged array across `@odata.nextLink`
- `getJson($path)` — GET and decode single object
- `isUuid($value)` — simple UUID v4/v5 check to branch logic

You can also target nested resources by composing paths in your methods, e.g. `/groups/{id}/members`, `/users/{id}/manager`, etc.

---

### End-to-end examples

- List users with selective fields:

```php
$users = new AmgGroup\UsersClient($api);

$list = $users->getGroups('alex@example.com', GraphQuery::create()->select('id,displayName'));
```

- Find a Conditional Access policy and disable it:

```php
$ca = new AmgGroup\ConditionalAccessClient($api);

$policy = $ca->find('Require MFA for admins');
if ($policy) {
    $ca->update($policy['id'], ['state' => 'disabled']);
}
```

- Create a new group using your custom `GroupsClient` (from above):

```php
$groups = new App\Graph\GroupsClient($api);

$resp = $groups->create([
    'displayName'    => 'Marketing',
    'mailEnabled'    => false,
    'mailNickname'   => 'marketing',
    'securityEnabled'=> true,
    'groupTypes'     => [],
]);

if ($resp && $resp->getStatusCode() >= 200 && $resp->getStatusCode() < 300) {
    $group = json_decode((string)$resp->getBody(), true);
}
```

---

### Error handling tips

- Most write operations return `ResponseInterface`. Always check status codes and parse the body for error details from Graph.
- For consistency-sensitive queries (e.g., `$count`, advanced filters), add header `ConsistencyLevel: eventual`.
- For high-volume list operations, enable backoff to avoid throttling (HTTP 429).

---

### Project Structure (key classes)

- `MSGraph` — base class for resource clients (retry/backoff, pagination, helpers)
- `UsersClient` — `findUser`, `getGroups`, `getMember`
- `ConditionalAccessClient` — `list`, `find`, `create`, `update`, `delete`
- `GraphQuery` — builds `$select`, `$filter`, `$top`, `$orderby`, `$expand`
- `Filter` — helper to compose filter expressions (`eq`, `ne`, `contains`, `startsWith`, `and`, `or`, `not`, `in`)

---

### Versioning and Graph beta

You control Graph API version via the `base_uri` you pass to `ApiClient` (v1.0 or beta). You can even instantiate two `ApiClient`s if you need to call both versions from the same process.

```php
$apiV1  = new ApiClient(['base_uri' => 'https://graph.microsoft.com/v1.0', 'auth' => $tokenProvider]);
$apiBeta= new ApiClient(['base_uri' => 'https://graph.microsoft.com/beta',  'auth' => $tokenProvider]);

$usersV1  = new AmgGroup\UsersClient($apiV1);
$usersB   = new AmgGroup\UsersClient($apiBeta);
```

---

### Contributing / Extending

- Create small, single-responsibility clients per Graph resource.
- Prefer `GraphQuery` + `Filter` to keep OData logic readable.
- Use `enableBackoff()` in production scenarios where throttling is likely.

PRs for additional resource clients (e.g., `GroupsClient`, `ServicePrincipalsClient`, `DevicesClient`) are welcome.

---

### License

This project is provided by AMG Group. See `composer.json` for package details.
