<?php

namespace AmgGroup;

/**
 * GraphQuery
 * ----------
 * Helper for building OData query parameters for Microsoft Graph.
 */
final class GraphQuery
{
    private array $select = [];
    private ?string $filter = null;
    private ?int $top = null;
    private array $orderBy = [];
    private array $expand = [];

    public static function create(): self
    {
        return new self();
    }

    public function select(string|array $fields): self
    {
        $this->select = is_array($fields) ? $fields : array_map('trim', explode(',', $fields));
        return $this;
    }

    public function filter(string $filter): self
    {
        $this->filter = $filter;
        return $this;
    }

    public function top(int $top): self
    {
        $this->top = $top;
        return $this;
    }

    public function orderBy(string|array $order): self
    {
        $this->orderBy = is_array($order) ? $order : array_map('trim', explode(',', $order));
        return $this;
    }

    public function expand(string|array $relationships): self
    {
        $this->expand = is_array($relationships) ? $relationships : array_map('trim', explode(',', $relationships));
        return $this;
    }

    public function toArray(): array
    {
        $params = [];
        if ($this->select) {
            $params['$select'] = implode(',', $this->select);
        }
        if ($this->filter !== null && $this->filter !== '') {
            $params['$filter'] = $this->filter;
        }
        if ($this->top !== null) {
            $params['$top'] = $this->top;
        }
        if ($this->orderBy) {
            $params['$orderby'] = implode(',', $this->orderBy);
        }
        if ($this->expand) {
            $params['$expand'] = implode(',', $this->expand);
        }
        return $params;
    }

    public function toQueryString(): string
    {
        $arr = $this->toArray();
        return http_build_query($arr);
    }
}
