<?php

namespace AmgGroup;

use Psr\Http\Message\ResponseInterface;

final class ConditionalAccessClient extends MSGraph
{
    private string $path = '/identity/conditionalAccess/policies';

    /** List all policies, optionally reindexed. Supports typed query for $select/$filter/etc. */
    public function list(?GraphQuery $q = null, string $indexBy = 'id'): array
    {
        $params = $q ? $q->toArray() : [];
        $items = $this->getAllPages($this->path, $params);
        if ($indexBy) {
            $indexed = [];
            foreach ($items as $it) {
                if (is_array($it) && array_key_exists($indexBy, $it)) {
                    $indexed[$it[$indexBy]] = $it;
                }
            }
            return $indexed;
        }
        return $items;
    }

    /** Find a policy by UUID or by displayName. */
    public function find(string $ref, ?GraphQuery $q = null): ?array
    {
        if ($this->isUuid($ref)) {
            return $this->getJson($this->path . '/' . $ref);
        }
        $query = $q ?? GraphQuery::create();
        $query->filter(Filter::eq('displayName', $ref))->top(2);
        $items = $this->getAllPages($this->path, $query->toArray());
        if (count($items) === 1) {
            return $items[0];
        }
        return null;
    }

    public function create(array $policy): ?ResponseInterface
    {
        return $this->request('POST', $this->path, ['json' => $policy]);
    }

    public function update(string $ref, array $patch): ?ResponseInterface
    {
        $id = $this->resolvePolicyId($ref);
        if ($id === null) {
            return null;
        }
        return $this->request('PATCH', $this->path . '/' . $id, ['json' => $patch]);
    }

    public function delete(string $ref): ?ResponseInterface
    {
        $id = $this->resolvePolicyId($ref);
        if ($id === null) {
            return null;
        }
        return $this->request('DELETE', $this->path . '/' . $id);
    }

    private function resolvePolicyId(string $ref): ?string
    {
        if ($this->isUuid($ref)) {
            return $ref;
        }
        $found = $this->find($ref);
        return $found['id'] ?? null;
    }
}
