<?php

declare(strict_types=1);

namespace AmgGroup\Tests;

use AmgGroup\GraphQuery;
use AmgGroup\UsersClient;
use AmgGroup\ConfigInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;

final class UsersClientTest extends TestCase
{
    private function makeConfigStub(): ConfigInterface
    {
        return new class implements ConfigInterface {
            public function get(string $key, mixed $default = null): mixed { return $default; }
            public function set(string $key, mixed $value): void {}
            public function isEmpty(): bool { return true; }
            public function getLoadedPaths(): array { return []; }
            public function getSkippedPaths(): array { return []; }
        };
    }

    private function makeTokenProvider(Client $http): \AmgGroup\MsGraphClientCredentialsTokenProvider
    {
        return new \AmgGroup\MsGraphClientCredentialsTokenProvider(
            $http,
            tenantId: 'common',
            clientId: 'client',
            clientSecret: 'secret'
        );
    }

    public function testFindUserByUuidReturnsSingleUser(): void
    {
        $uuid = '3fa85f64-5717-4562-b3fc-2c963f66afa6';
        $body = json_encode(['id' => $uuid, 'displayName' => 'Alice']);
        $mock = new MockHandler([
            // token
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            // graph
            new Response(200, ['Content-Type' => 'application/json'], $body),
        ]);
        $stack = HandlerStack::create($mock);
        $history = [];
        $stack->push(Middleware::history($history));
        $http = new Client(['handler' => $stack]);

        $client = new UsersClient(
            $http,
            new NullLogger(),
            $this->makeTokenProvider($http),
            $this->makeConfigStub(),
            apiVersion: 'v1.0'
        );

        $client->findUser($uuid);
        // Verify the request path
        $this->assertCount(2, $history); // token + api call
        $this->assertStringContainsString('/v1.0/users/' . $uuid, (string)$history[1]['request']->getUri());
    }

    public function testFindUserByUpnWithTop2AndFilter(): void
    {
        $page = json_encode(['value' => [
            ['id' => 'u1', 'userPrincipalName' => 'alice@example.com']
        ]]);
        $mock = new MockHandler([
            // token
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            // graph
            new Response(200, ['Content-Type' => 'application/json'], $page),
        ]);
        $stack = HandlerStack::create($mock);
        $history = [];
        $stack->push(Middleware::history($history));
        $http = new Client(['handler' => $stack]);

        $client = new UsersClient($http, new NullLogger(), $this->makeTokenProvider($http), $this->makeConfigStub());

        $user = $client->findUser('alice@example.com');
        $this->assertNotNull($user);
        $this->assertSame('u1', $user['id']);

        // Assert query contains $filter and $top=2
        $this->assertCount(2, $history);
        $uri = (string)$history[1]['request']->getUri();
        $this->assertStringContainsString('/v1.0/users', $uri);
        $this->assertStringContainsString('%24top=2', $uri);
        $this->assertStringContainsString('%24filter=', $uri);
    }

    public function testFindUserAmbiguousReturnsNull(): void
    {
        $page = json_encode(['value' => [
            ['id' => 'u1'], ['id' => 'u2']
        ]]);
        $mock = new MockHandler([
            // token
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            // graph
            new Response(200, ['Content-Type' => 'application/json'], $page),
        ]);
        $http = new Client(['handler' => HandlerStack::create($mock)]);

        $client = new UsersClient($http, new NullLogger(), $this->makeTokenProvider($http), $this->makeConfigStub());

        $user = $client->findUser('Alice Smith');
        $this->assertNull($user);
    }

    public function testGetGroupsByUserId(): void
    {
        $uuid = '3fa85f64-5717-4562-b3fc-2c963f66afa6';
        $page = json_encode(['value' => [
            ['id' => 'g1'], ['id' => 'g2']
        ]]);
        $mock = new MockHandler([
            // token
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            // resolve user by UUID – no lookup needed, then /memberOf listing
            new Response(200, ['Content-Type' => 'application/json'], $page),
        ]);
        $stack = HandlerStack::create($mock);
        $history = [];
        $stack->push(Middleware::history($history));
        $http = new Client(['handler' => $stack]);

        $client = new UsersClient($http, new NullLogger(), $this->makeTokenProvider($http), $this->makeConfigStub());

        $client->getGroups($uuid, GraphQuery::create()->select(['id']));

        $this->assertStringContainsString('/v1.0/users/' . $uuid . '/memberOf', (string)$history[1]['request']->getUri());
        $this->assertStringContainsString('%24select=id', (string)$history[1]['request']->getUri());
    }
}
