<?php

declare(strict_types=1);

namespace AmgGroup\Tests;

use AmgGroup\ConditionalAccessClient;
use AmgGroup\ConfigInterface;
use AmgGroup\GraphQuery;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;

final class ConditionalAccessClientTest extends TestCase
{
    private function makeConfigStub(): ConfigInterface
    {
        return new class implements ConfigInterface {
            public function get(string $key, mixed $default = null): mixed { return $default; }
            public function set(string $key, mixed $value): void {}
            public function isEmpty(): bool { return true; }
            public function getLoadedPaths(): array { return []; }
            public function getSkippedPaths(): array { return []; }
        };
    }

    private function makeTokenProvider(Client $http): \AmgGroup\MsGraphClientCredentialsTokenProvider
    {
        return new \AmgGroup\MsGraphClientCredentialsTokenProvider(
            $http,
            tenantId: 'common',
            clientId: 'client',
            clientSecret: 'secret'
        );
    }

    public function testListPoliciesIndexedById(): void
    {
        $page = json_encode(['value' => [
            ['id' => 'p1', 'displayName' => 'Policy 1'],
            ['id' => 'p2', 'displayName' => 'Policy 2'],
        ]]);
        $mock = new MockHandler([
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            new Response(200, ['Content-Type' => 'application/json'], $page),
        ]);
        $stack = HandlerStack::create($mock);
        $history = [];
        $stack->push(Middleware::history($history));
        $http = new Client(['handler' => $stack]);

        $client = new ConditionalAccessClient($http, new NullLogger(), $this->makeTokenProvider($http), $this->makeConfigStub(), apiVersion: 'beta');

        $items = $client->list(GraphQuery::create()->select(['id','displayName']), indexBy: 'id');
        $this->assertArrayHasKey('p1', $items);
        $this->assertSame('Policy 1', $items['p1']['displayName']);

        $this->assertStringContainsString('/beta/identity/conditionalAccess/policies', (string)$history[1]['request']->getUri());
        $this->assertStringContainsString('%24select=id%2CdisplayName', (string)$history[1]['request']->getUri());
    }

    public function testFindByNameAndUpdateByResolvedId(): void
    {
        $listPage = json_encode(['value' => [ ['id' => 'pid-123', 'displayName' => 'Require MFA'] ]]);
        $mock = new MockHandler([
            // token
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            // find (list with filter)
            new Response(200, ['Content-Type' => 'application/json'], $listPage),
            // update(): resolve id via find again
            new Response(200, ['Content-Type' => 'application/json'], $listPage),
            // update PATCH
            new Response(204, []),
        ]);
        $stack = HandlerStack::create($mock);
        $history = [];
        $stack->push(Middleware::history($history));
        $http = new Client(['handler' => $stack]);

        $client = new ConditionalAccessClient($http, new NullLogger(), $this->makeTokenProvider($http), $this->makeConfigStub());

        $found = $client->find('Require MFA');
        $this->assertNotNull($found);
        $this->assertSame('pid-123', $found['id']);

        $client->update('Require MFA', ['state' => 'enabled']);

        // Fourth request should be PATCH to .../policies/pid-123
        $this->assertSame('PATCH', $history[3]['request']->getMethod());
        $this->assertStringContainsString('/v1.0/identity/conditionalAccess/policies/pid-123', (string)$history[3]['request']->getUri());
    }

    public function testDeleteByNameResolvesToId(): void
    {
        $listPage = json_encode(['value' => [ ['id' => 'pid-xyz', 'displayName' => 'Legacy Policy'] ]]);
        $mock = new MockHandler([
            // token
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            // delete(): resolve via find
            new Response(200, ['Content-Type' => 'application/json'], $listPage),
            // delete call
            new Response(204, []),
        ]);
        $stack = HandlerStack::create($mock);
        $history = [];
        $stack->push(Middleware::history($history));
        $http = new Client(['handler' => $stack]);

        $client = new ConditionalAccessClient($http, new NullLogger(), $this->makeTokenProvider($http), $this->makeConfigStub());

        $client->delete('Legacy Policy');
        $this->assertSame('DELETE', $history[2]['request']->getMethod());
        $this->assertStringContainsString('/v1.0/identity/conditionalAccess/policies/pid-xyz', (string)$history[2]['request']->getUri());
    }
}
