<?php

declare(strict_types=1);

namespace AmgGroup\Tests;

use AmgGroup\ConfigInterface;
use AmgGroup\MSGraph;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;

final class MSGraphBackoffTest extends TestCase
{
    private function makeConfigStub(): ConfigInterface
    {
        return new class implements ConfigInterface {
            public function get(string $key, mixed $default = null): mixed { return $default; }
            public function set(string $key, mixed $value): void {}
            public function isEmpty(): bool { return true; }
            public function getLoadedPaths(): array { return []; }
            public function getSkippedPaths(): array { return []; }
        };
    }

    private function makeTokenProvider(Client $http): \AmgGroup\MsGraphClientCredentialsTokenProvider
    {
        return new \AmgGroup\MsGraphClientCredentialsTokenProvider(
            $http,
            tenantId: 'common',
            clientId: 'client',
            clientSecret: 'secret'
        );
    }

    public function testBackoffRetriesOn429WhenHttpErrorsDisabled(): void
    {
        // Queue: token, 429, 200
        $mock = new MockHandler([
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['access_token' => 't', 'expires_in' => 3600])),
            new Response(429, ['Retry-After' => '0']),
            new Response(200, ['Content-Type' => 'application/json'], json_encode(['ok' => true])),
        ]);
        $stack = HandlerStack::create($mock);
        $history = [];
        $stack->push(Middleware::history($history));
        $http = new Client(['handler' => $stack]);

        // Create a small concrete client to expose protected request()
        $client = new class(
            $http,
            new NullLogger(),
            $this->makeTokenProvider($http),
            $this->makeConfigStub()
        ) extends MSGraph {
            public function doRequest(string $method, string $path, array $options = []) { return $this->request($method, $path, $options); }
        };

        $client->enableBackoff(2, 1);

        $resp = $client->doRequest('GET', '/me', ['http_errors' => false]);
        $this->assertNotNull($resp);
        $this->assertSame(200, $resp->getStatusCode());

        // 3 total requests: token + 429 + retry 200
        $this->assertCount(3, $history);
        $this->assertSame(200, $client->getStatusCode());
    }
}
