<?php

namespace AmgGroup;

use Psr\Http\Message\ResponseInterface;

final class UsersClient extends MSGraph
{
    private string $path = '/users';

    /**
     * Find a user by UUID or by name/UPN/mail.
     * Returns the user object as array or null if not found/ambiguous.
     */
    public function findUser(string $ref, ?GraphQuery $q = null): ?array
    {
        if ($this->isUuid($ref)) {
            return $this->getJson($this->path . '/' . $ref);
        }

        $query = $q ?? GraphQuery::create();
        $filter = Filter::or(
            Filter::eq('userPrincipalName', $ref),
            Filter::eq('mail', $ref),
            Filter::eq('displayName', $ref)
        );
        $query->filter($filter)->top(2);

        $items = $this->getAllPages($this->path, $query->toArray());
        if (count($items) === 1) {
            return $items[0];
        }
        return null; // not found or ambiguous
    }

    /**
     * Get all groups a user is a member of.
     * Returns an array (possibly empty) of directoryObjects.
     */
    public function getGroups(string $userRef, ?GraphQuery $q = null): array
    {
        $userId = $this->resolveUserId($userRef);
        if ($userId === null) {
            return [];
        }
        $params = $q ? $q->toArray() : [];
        return $this->getAllPages($this->path . '/' . $userId . '/memberOf', $params);
    }

    /**
     * Get a specific member under a group by UUID or name. Returns array|null.
     * This is a convenience for GET /groups/{groupId}/members/{memberId}
     */
    public function getMember(string $groupRef, string $memberRef): ?array
    {
        $groupId = $this->resolveGroupId($groupRef);
        $memberId = $this->resolveUserId($memberRef);
        if ($groupId === null || $memberId === null) {
            return null;
        }
        return $this->getJson('/groups/' . $groupId . '/members/' . $memberId);
    }

    // ---- Helpers ----
    private function resolveUserId(string $ref): ?string
    {
        if ($this->isUuid($ref)) {
            return $ref;
        }
        $found = $this->findUser($ref);
        return $found['id'] ?? null;
    }

    private function resolveGroupId(string $ref): ?string
    {
        if ($this->isUuid($ref)) {
            return $ref;
        }
        // Try displayName exact match, then mailNickname
        $query = GraphQuery::create()->filter(
            Filter::or(
                Filter::eq('displayName', $ref),
                Filter::eq('mailNickname', $ref)
            )
        )->top(2);
        $items = $this->getAllPages('/groups', $query->toArray());
        if (count($items) === 1) {
            return $items[0]['id'] ?? null;
        }
        return null;
    }
}
