<?php

namespace AmgGroup;

use GuzzleHttp\ClientInterface as GuzzleClient;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;

/**
 * MSGraphBase
 * -----------
 * A small extensible base for Microsoft Graph resource clients.
 *
 * - Per-instance API version (v1.0/beta)
 * - DI-friendly: accepts Guzzle client, logger, token provider, config
 * - Optional retry/backoff for 429/5xx
 * - Exposes ApiClient helpers and OData query utilities
 */
abstract class MSGraph
{
    protected ApiClient $api;

    protected string $graphApiVersion;
    protected string $graphBase;

    // Backoff config
    private bool $backoffEnabled = false;
    private int $maxRetries = 3;
    private int $baseDelayMs = 250; // initial backoff

    public function __construct(
        GuzzleClient $http,
        LoggerInterface $logger,
        MsGraphClientCredentialsTokenProvider $tokenProvider,
        ConfigInterface $config,
        string $apiVersion = 'v1.0',
        string $graphBase = 'https://graph.microsoft.com'
    ) {
        $this->graphApiVersion = $apiVersion;
        $this->graphBase = rtrim($graphBase, '/');

        $baseUri = $this->graphBase . '/' . $this->graphApiVersion;
        $this->api = new ApiClient(
            http: $http,
            logger: $logger,
            tokenProvider: $tokenProvider,
            config: $config,
            baseUri: $baseUri,
            defaultHeaders: [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ]
        );
    }

    // ---- Backoff controls ----
    public function enableBackoff(int $maxRetries = 3, int $baseDelayMs = 250): static
    {
        $this->backoffEnabled = true;
        $this->maxRetries = max(0, $maxRetries);
        $this->baseDelayMs = max(1, $baseDelayMs);
        return $this;
    }

    public function disableBackoff(): static
    {
        $this->backoffEnabled = false;
        return $this;
    }

    // ---- Helpers ----
    protected function url(string $path): string
    {
        return '/' . ltrim($path, '/');
    }

    protected function resource(string $path): ResourceClient
    {
        return new ResourceClient($this->api, $this->url($path));
    }

    /** Convenience pass-through with optional retry/backoff */
    protected function request(string $method, string $path, array $options = []): ?ResponseInterface
    {
        if (!$this->backoffEnabled) {
            return $this->api->request($method, $this->url($path), $options);
        }

        $attempt = 0;
        $response = $this->api->request($method, $this->url($path), $options);
        while ($attempt < $this->maxRetries && $response) {
            $status = $response->getStatusCode();
            if (!in_array($status, [429, 500, 502, 503, 504], true)) {
                break;
            }
            $attempt++;
            $delayMs = $this->computeDelay($response, $attempt);
            usleep($delayMs * 1000);
            $response = $this->api->request($method, $this->url($path), $options);
        }
        return $response;
    }

    protected function getAllPages(string $path, array $query = []): array
    {
        $q = $query ? ('?' . http_build_query($query)) : '';
        return $this->api->getAllPages($this->url($path) . $q);
    }

    public function setDefaultHeaders(array $headers): static
    {
        $this->api->setDefaultHeaders($headers);
        return $this;
    }

    public function getLastResponse(): ?ResponseInterface
    {
        return $this->api->getLastResponse();
    }

    public function getStatusCode(): int
    {
        return $this->api->getStatusCode();
    }

    /** Decode JSON for single-object GET endpoints; returns null on failure */
    protected function getJson(string $path): ?array
    {
        $resp = $this->api->request('GET', $this->url($path));
        if (!$resp) {
            return null;
        }
        $data = json_decode((string)$resp->getBody(), true);
        return is_array($data) ? $data : null;
    }

    protected function isUuid(string $value): bool
    {
        return (bool)preg_match(
            '/^[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[1-5][0-9a-fA-F]{3}-[89abAB][0-9a-fA-F]{3}-[0-9a-fA-F]{12}$/',
            $value
        );
    }

    private function computeDelay(ResponseInterface $response, int $attempt): int
    {
        $retryAfter = $response->getHeaderLine('Retry-After');
        if ($retryAfter !== '') {
            if (ctype_digit($retryAfter)) {
                return max(0, (int)$retryAfter * 1000);
            }
            $ts = strtotime($retryAfter);
            if ($ts !== false) {
                $delta = max(0, $ts - time());
                return $delta * 1000;
            }
        }
        // exponential backoff with jitter
        $base = $this->baseDelayMs * (2 ** ($attempt - 1));
        $jitter = random_int(0, (int)floor($base * 0.2));
        return $base + $jitter;
    }
}