<?php
namespace AmgGroup\Tests;

use PHPUnit\Framework\TestCase;
use AmgGroup\MSConditionalAccess;
use ReflectionClass;
use ReflectionMethod;

class MSConditionalAccessNamedLocationsTest extends TestCase
{
    protected function setUp(): void
    {
        // Reset session
        if (session_status() !== PHP_SESSION_NONE) {
            session_destroy();
        }
        session_start();
        $_SESSION = [];

        // Reset MSGraphBase singleton instances store
        $baseRef = new ReflectionClass('AmgGroup\\MSGraphBase');
        $instancesProp = $baseRef->getProperty('instances');
        $instancesProp->setAccessible(true);
        $instancesProp->setValue([]);
    }

    public function testGetNamedLocationsReturnsTrueValues(): void
    {
        $conditionalAccess = MSConditionalAccess::getInstance();

        // Seed the Locations cache directly via ArrayAccess to avoid external API calls
        $locations = [
            [
                'id' => 'loc-1',
                'displayName' => 'Headquarters',
                'ipRanges' => ['1.2.3.4/32'],
                'note' => 'Main office'
            ],
            [
                'id' => 'loc-2',
                'displayName' => 'Branch',
                'countriesAndRegions' => ['US', 'CA'],
                'includeUnknownCountriesAndRegions' => false
            ],
            [
                'id' => 'loc-3',
                'displayName' => '', // Should be ignored because displayName is empty
                'ipRanges' => ['5.6.7.8/32']
            ],
            [
                'id' => 'loc-4',
                // No displayName -> should be ignored
                'ipRanges' => ['9.9.9.0/24']
            ],
        ];

        // Store in cache under the expected dataset key
        $conditionalAccess['Locations'] = $locations;

        // Access the protected getNamedLocations() via reflection
        $method = new ReflectionMethod(MSConditionalAccess::class, 'getNamedLocations');
        $method->setAccessible(true);
        $result = $method->invoke($conditionalAccess);

        // Ensure the result maps displayName => full location array ("true values")
        $this->assertIsArray($result);
        $this->assertArrayHasKey('Headquarters', $result);
        $this->assertArrayHasKey('Branch', $result);
        $this->assertSame($locations[0], $result['Headquarters']);
        $this->assertSame($locations[1], $result['Branch']);

        // Ensure entries without a valid displayName are not present
        $this->assertArrayNotHasKey('', $result);
        $this->assertCount(2, $result);
    }
}
