<?php
declare(strict_types=1);

namespace AmgGroup;

class MSUsers extends MSGraphBase
{
    const CACHE_KEY = 'msusers';

    protected array $cacheDefaults = [
        'Users' => 300,
        'ActiveUsers' => 300
    ];

    protected function getUsers(): array
    {
        $excludeUsers = $this->config->get('excludeUsers');
        $api = ApiClient::getInstance();
        $users = $api->getAllPages(
            url: 'https://graph.microsoft.com/v1.0/users',
            params: ['$select' => 'displayName,accountEnabled,givenName,surname,jobTitle,mail,officeLocation,userPrincipalName,id']
        );

        $userList = [];
        foreach ($users as $user) {
            if (!isset($excludeUsers[$user['id']])) {
                $userList[$user['id']] = $user;
            }
        }
        return $userList;
    }

    protected function getActiveUsers(): array
    {
        $users = $this['Users'];
        if (empty($users)) {
            throw new \Exception('No users found');
        }

        $activeUsers = [];
        foreach ($users as $user) {
            if (isset($user['accountEnabled']) && $user['accountEnabled']) {
                $activeUsers[$user['id']] = $user;
            }
        }
        return $activeUsers;
    }

    // Implement write-back functionality
    protected function writeToApi(string $dataSet, string $id, array $change): bool
    {
        $api = ApiClient::getInstance();

        switch ($change['operation']) {
            case 'update':
                // Update user via PATCH request
                $response = $api->request(
                    'PATCH',
                    "https://graph.microsoft.com/v1.0/users/{$id}",
                    ['json' => $change['newValue']]
                );
                return $response && $response->getStatusCode() === 204;

            case 'delete':
                // Delete user via DELETE request
                $response = $api->request(
                    'DELETE',
                    "https://graph.microsoft.com/v1.0/users/{$id}"
                );
                return $response && $response->getStatusCode() === 204;

            case 'create':
                // Create user via POST request
                $response = $api->request(
                    'CREATE',
                    "https://graph.microsoft.com/v1.0/users",
                    ['json' => $change['newValue']]
                );
                return $response && in_array($response->getStatusCode(), [200, 201]);

            default:
                $this->logger->error("Unknown operation: {$change['operation']}");
                return false;
        }
    }
}