<?php

namespace AmgGroup;

class MSConditionalAccess extends MSGraphBase
{
    const CACHE_KEY = 'conditionalaccess';

    protected array $cacheDefaults = [
        'Locations' => 300,
        'Policies' => 300,
    ];

    /**
     * Get all named locations indexed by 'id'
     *
     * @return array
     */
    protected function getLocations(): array
    {
        $api = ApiClient::getInstance();
        $namedLocations = $api->getAllPages(url: 'https://graph.microsoft.com/v1.0/identity/conditionalAccess/namedLocations');
        $locations = [];
        foreach ($namedLocations as $location) {
            $locations[$location['id']] = $location;
        }
        return $locations;
    }

    protected function getNamedLocations(): array
    {
        $locations = $this['Locations'];
        if (empty($locations)) {
            throw new \Exception('No locations found');

        }
        $namedLocations = [];
        foreach ($locations as $location) {
            if (isset($location['displayName']) && !empty($location['displayName'])) {
                $namedLocations[$location['displayName']] = $location;
            }
        }
        return $namedLocations;
    }

    protected function getPolicies(): array
    {
        $api = ApiClient::getInstance();
        $policies = $api->getAllPages(url: 'https://graph.microsoft.com/v1.0/identity/conditionalAccess/policies');
        $policiesList = [];
        foreach ($policies as $policy) {
            $policiesList[$policy['id']] = $policy;
        }
        return $policiesList;
    }

    protected function getNamedPolicies(): array
    {
        $policies = $this['Policies'];
        if (empty($policies)) {
            throw new \Exception('No policies found');
        }
        $namedPolicies = [];
        foreach ($policies as $policy) {
            if (isset($policy['displayName']) && !empty($policy['displayName'])) {
                $namedPolicies[$policy['displayName']] = $policy;
            }
        }
        return $namedPolicies;

    }

    // Implement write-back functionality
    protected function writeToApi(string $dataSet, string $id, array $change): bool
    {
        $api = ApiClient::getInstance();

        switch ($change['operation']) {
            case 'update':
                // Update user via PATCH request
                $response = $api->request(
                    'PATCH',
                    "https://graph.microsoft.com/v1.0/identity/conditionalAccess/policies/{$id}",
                    ['json' => $change['newValue']]
                );
                return $response && $response->getStatusCode() === 204;

            case 'delete':
                // Delete user via DELETE request
                $response = $api->request(
                    'DELETE',
                    "https://graph.microsoft.com/v1.0/identity/conditionalAccess/policies/{$id}"
                );
                return $response && $response->getStatusCode() === 204;

            case 'create':
                // Create user via POST request
                $response = $api->request(
                    'CREATE',
                    "https://graph.microsoft.com/v1.0/identity/conditionalAccess/policies",
                    ['json' => $change['newValue']]
                );
                return $response && in_array($response->getStatusCode(), [200, 201]);

            default:
                $this->logger->error("Unknown operation: {$change['operation']}");
                return false;
        }
    }

    public function invalidatePoliciesCache(): void
    {
        $this->clearCache('Policies');
    }
    public function invalidateNamedPoliciesCache(): void
    {
        $this->clearCache('NamedPolicies');
    }
    public function invalidateLocationsCache(): void
    {
        $this->clearCache('Locations');
    }
    public function invalidateNamedLocationsCache(): void
    {
        $this->clearCache('NamedLocations');
    }
}