<?php

namespace AmgGroup;

abstract class MSGraphBase implements \ArrayAccess
{
    protected Config $config;
    protected Logger $logger;
    protected array $cacheDefaults = [];

    // Change tracking for write-back functionality
    protected array $dirtyData = [];      // Track what has changed
    protected array $originalData = [];   // Keep original values for comparison

    // Static instances array to store singleton instances per class
    private static array $instances = [];

    protected function __construct()
    {
        $this->config = Config::getInstance();
        $this->logger = Logger::getInstance();

        \AmgGroup\Session::start();
        if (!isset($_SESSION[static::CACHE_KEY])) {
            $_SESSION[static::CACHE_KEY] = [];
        }
    }

    // Generic getInstance method for all subclasses
    public static function getInstance(): static
    {
        $class = static::class;

        if (!isset(self::$instances[$class])) {
            self::$instances[$class] = new static();
        }

        return self::$instances[$class];
    }

    // All ArrayAccess methods in base class
    public function offsetExists(mixed $offset): bool
    {
        return $this->getFromSession($offset, static::CACHE_KEY) !== null;
    }

    public function offsetGet(mixed $offset): mixed
    {
        $cached = $this->getFromSession($offset, static::CACHE_KEY);
        if ($cached !== null) {
            // Store original data for change tracking
            if (!isset($this->originalData[$offset])) {
                $this->originalData[$offset] = $cached;
            }
            return new WritableDataArray($cached, $this, $offset);
        }

        $fetchedData = $this->fetchFromApi($offset);
        $this->storeInSession($offset, $fetchedData, static::CACHE_KEY);

        // Store original data for change tracking
        $this->originalData[$offset] = $fetchedData;

        return new WritableDataArray($fetchedData, $this, $offset);
    }

    public function offsetSet(mixed $offset, mixed $value): void
    {
        $this->storeInSession($offset, $value, static::CACHE_KEY);
        $this->markDirty($offset, null, $value);
    }

    public function offsetUnset(mixed $offset): void
    {
        unset($_SESSION[static::CACHE_KEY][$offset]);
        $this->markDirty($offset, null, null, 'delete');
    }

    // Change tracking methods
    public function markDirty(string $dataSet, ?string $id, mixed $newValue, string $operation = 'update'): void
    {
        if (!isset($this->dirtyData[$dataSet])) {
            $this->dirtyData[$dataSet] = [];
        }

        $this->dirtyData[$dataSet][$id ?? 'root'] = [
            'operation' => $operation,
            'newValue' => $newValue,
            'timestamp' => time()
        ];

        $this->logger->debug("Marked {$dataSet}[{$id}] as dirty", ['operation' => $operation]);
    }

    public function flush(string $dataSet = null): bool
    {
        $success = true;
        $datasetsToFlush = $dataSet ? [$dataSet] : array_keys($this->dirtyData);

        foreach ($datasetsToFlush as $dataset) {
            if (!isset($this->dirtyData[$dataset])) {
                continue;
            }

            foreach ($this->dirtyData[$dataset] as $id => $change) {
                try {
                    $result = $this->writeToApi($dataset, $id, $change);
                    if ($result) {
                        // Update the cached data with the new value
                        if ($change['operation'] === 'update') {
                            $this->updateCachedData($dataset, $id, $change['newValue']);
                        } elseif ($change['operation'] === 'delete') {
                            $this->removeCachedData($dataset, $id);
                        }

                        $this->logger->info("Successfully wrote {$dataset}[{$id}] to API");
                    } else {
                        $success = false;
                        $this->logger->error("Failed to write {$dataset}[{$id}] to API");
                    }
                } catch (\Exception $e) {
                    $success = false;
                    $this->logger->error("Exception writing {$dataset}[{$id}] to API", ['error' => $e->getMessage()]);
                }
            }

            // Clear dirty data for this dataset if successful
            if ($success) {
                unset($this->dirtyData[$dataset]);
            }
        }

        return $success;
    }

    public function hasPendingChanges(string $dataSet = null): bool
    {
        if ($dataSet) {
            return isset($this->dirtyData[$dataSet]) && !empty($this->dirtyData[$dataSet]);
        }
        return !empty($this->dirtyData);
    }

    public function getPendingChanges(): array
    {
        return $this->dirtyData;
    }

    public function rollback(string $dataSet = null): void
    {
        $datasetsToRollback = $dataSet ? [$dataSet] : array_keys($this->dirtyData);

        foreach ($datasetsToRollback as $dataset) {
            if (isset($this->originalData[$dataset])) {
                $this->storeInSession($dataset, $this->originalData[$dataset], static::CACHE_KEY);
                unset($this->dirtyData[$dataset]);
                $this->logger->info("Rolled back changes for {$dataset}");
            }
        }
    }

    // Abstract method for concrete classes to implement
    abstract protected function writeToApi(string $dataSet, string $id, array $change): bool;

    // Cache management methods
    protected function getFromSession(string $offset, string $cacheKey): mixed
    {
        if (isset($_SESSION[$cacheKey][$offset])) {
            $entry = $_SESSION[$cacheKey][$offset];
            $ttl = $this->cacheDefaults[$offset] ?? 300;
            if (time() - $entry['timestamp'] <= $ttl) {
                return $entry['data'];
            }
            unset($_SESSION[$cacheKey][$offset]);
        }
        return null;
    }

    protected function storeInSession(string $offset, mixed $value, string $cacheKey): void
    {
        $_SESSION[$cacheKey][$offset] = [
            'data' => $value,
            'timestamp' => time()
        ];
    }

    protected function updateCachedData(string $dataSet, string $id, mixed $newValue): void
    {
        if (isset($_SESSION[static::CACHE_KEY][$dataSet])) {
            if ($id === 'root') {
                $_SESSION[static::CACHE_KEY][$dataSet]['data'] = $newValue;
            } else {
                $_SESSION[static::CACHE_KEY][$dataSet]['data'][$id] = $newValue;
            }
            $_SESSION[static::CACHE_KEY][$dataSet]['timestamp'] = time();
        }
    }

    protected function removeCachedData(string $dataSet, string $id): void
    {
        if (isset($_SESSION[static::CACHE_KEY][$dataSet])) {
            if ($id === 'root') {
                unset($_SESSION[static::CACHE_KEY][$dataSet]);
            } else {
                unset($_SESSION[static::CACHE_KEY][$dataSet]['data'][$id]);
            }
        }
    }

    // Dynamic method calling in base class
    protected function fetchFromApi(string $offset): mixed
    {
        $methodName = 'get' . ucfirst($offset);

        if (!method_exists($this, $methodName)) {
            throw new \BadMethodCallException("Method {$methodName} does not exist in " . static::class);
        }

        return call_user_func([$this, $methodName]);
    }

    protected function clearCache($cacheType = null)
    {
        if ($cacheType) {
            unset($_SESSION[static::CACHE_KEY][$cacheType]);
        } else {
            unset($_SESSION[static::CACHE_KEY]);
        }
    }



}