<?php
namespace AmgGroup\Tests;

use AmgGroup\MSGraph;
use AmgGroup\Config;
use AmgGroup\Logger;
use AmgGroup\ApiClient;
use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\MockObject;

class MSGraphTest extends TestCase
{
    private MSGraph $msGraph;
    private MockObject $configMock;
    private MockObject $loggerMock;
    private MockObject $apiClientMock;

    protected function setUp(): void
    {
        // Clear any existing session data
        if (session_status() !== PHP_SESSION_NONE) {
            session_destroy();
        }

        // Start a new session for testing
        session_start();
        $_SESSION = [];

        // Reset the MSGraph singleton instance
        $this->resetSingleton(MSGraph::class);
        $this->resetSingleton(Config::class);
        $this->resetSingleton(Logger::class);

        // Create the MSGraph instance first
        $this->msGraph = MSGraph::getInstance();

        // Now mock the dependencies using reflection
        $this->injectMockedDependencies();
    }

    protected function tearDown(): void
    {
        if (session_status() !== PHP_SESSION_NONE) {
            session_destroy();
        }

        // Reset singletons
        $this->resetSingleton(MSGraph::class);
        $this->resetSingleton(Config::class);
        $this->resetSingleton(Logger::class);
    }

    private function resetSingleton(string $className): void
    {
        $reflection = new \ReflectionClass($className);
        $instanceProperty = $reflection->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue(null);
    }

    private function injectMockedDependencies(): void
    {
        $reflection = new \ReflectionClass($this->msGraph);

        // Mock and inject Config
        $this->configMock = $this->createMock(Config::class);
        $configProperty = $reflection->getProperty('config');
        $configProperty->setAccessible(true);
        $configProperty->setValue($this->msGraph, $this->configMock);

        // Mock and inject Logger
        $this->loggerMock = $this->createMock(Logger::class);
        $loggerProperty = $reflection->getProperty('logger');
        $loggerProperty->setAccessible(true);
        $loggerProperty->setValue($this->msGraph, $this->loggerMock);

        // Mock and inject ApiClient
        $this->apiClientMock = $this->createMock(ApiClient::class);
        $apiClientProperty = $reflection->getProperty('apiClient');
        $apiClientProperty->setAccessible(true);
        $apiClientProperty->setValue($this->msGraph, $this->apiClientMock);
    }

    public function testGetInstanceReturnsSameInstance(): void
    {
        $instance1 = MSGraph::getInstance();
        $instance2 = MSGraph::getInstance();

        $this->assertSame($instance1, $instance2);
        $this->assertInstanceOf(MSGraph::class, $instance1);
    }

    public function testGetAllUsersActiveUsers(): void
    {
        $mockUsers = [
            ['id' => 'user1', 'userPrincipalName' => 'user1@test.com', 'accountEnabled' => true],
            ['id' => 'user2', 'userPrincipalName' => 'user2@test.com', 'accountEnabled' => true]
        ];

        $this->configMock
            ->expects($this->once())
            ->method('get')
            ->with('excludeUsers')
            ->willReturn([]);

        $this->apiClientMock
            ->expects($this->once())
            ->method('getAllPages')
            ->with(
                'https://graph.microsoft.com/v1.0/users',
                ['$filter' => 'accountEnabled eq true']
            )
            ->willReturn($mockUsers);

        $this->loggerMock
            ->expects($this->atLeastOnce())
            ->method('debug');

        $this->loggerMock
            ->expects($this->once())
            ->method('info')
            ->with('Loading fresh data for usersUUID_active');

        $result = $this->msGraph->getAllUsers(true);

        $this->assertIsArray($result);
        $this->assertCount(2, $result);
        $this->assertArrayHasKey('user1', $result);
        $this->assertArrayHasKey('user2', $result);
    }

    public function testGetAllUsersAllUsers(): void
    {
        $mockUsers = [
            ['id' => 'user1', 'userPrincipalName' => 'user1@test.com', 'accountEnabled' => true],
            ['id' => 'user2', 'userPrincipalName' => 'user2@test.com', 'accountEnabled' => false]
        ];

        $this->configMock
            ->expects($this->once())
            ->method('get')
            ->with('excludeUsers')
            ->willReturn([]);

        $this->apiClientMock
            ->expects($this->once())
            ->method('getAllPages')
            ->with('https://graph.microsoft.com/v1.0/users', [])
            ->willReturn($mockUsers);

        $result = $this->msGraph->getAllUsers(false);

        $this->assertIsArray($result);
        $this->assertCount(2, $result);
    }

    public function testGetAllUsersWithExcludedUsers(): void
    {
        $mockUsers = [
            ['id' => 'user1', 'userPrincipalName' => 'user1@test.com'],
            ['id' => 'user2', 'userPrincipalName' => 'user2@test.com'],
            ['id' => 'excluded-user', 'userPrincipalName' => 'excluded@test.com']
        ];

        $this->configMock
            ->expects($this->once())
            ->method('get')
            ->with('excludeUsers')
            ->willReturn(['excluded-user' => true]);

        $this->apiClientMock
            ->expects($this->once())
            ->method('getAllPages')
            ->willReturn($mockUsers);

        $result = $this->msGraph->getAllUsers(true);

        $this->assertCount(2, $result);
        $this->assertArrayNotHasKey('excluded-user', $result);
    }

    public function testGetAllUsersByName(): void
    {
        $mockUsers = [
            ['id' => 'user1', 'userPrincipalName' => 'user1@test.com'],
            ['id' => 'user2', 'userPrincipalName' => 'user2@test.com']
        ];

        $this->configMock
            ->expects($this->once())
            ->method('get')
            ->with('excludeUsers')
            ->willReturn([]);

        $this->apiClientMock
            ->expects($this->once())
            ->method('getAllPages')
            ->willReturn($mockUsers);

        $result = $this->msGraph->getAllUsersByName(true);

        $this->assertIsArray($result);
        $this->assertArrayHasKey('user1@test.com', $result);
        $this->assertArrayHasKey('user2@test.com', $result);
        $this->assertEquals('user1', $result['user1@test.com']['id']);
    }

    public function testGetConditionalAccessPolicies(): void
    {
        $mockPolicies = [
            ['id' => 'policy1', 'displayName' => 'Policy 1'],
            ['id' => 'policy2', 'displayName' => 'Policy 2']
        ];

        $this->apiClientMock
            ->expects($this->once())
            ->method('getAllPages')
            ->with('https://graph.microsoft.com/v1.0/identity/conditionalAccess/policies')
            ->willReturn($mockPolicies);

        $result = $this->msGraph->getConditionalAccessPolicies();

        $this->assertIsArray($result);
        $this->assertCount(2, $result);
        $this->assertEquals('policy1', $result[0]['id']);
        $this->assertEquals('policy2', $result[1]['id']);
    }

    public function testGetConditionalAccessPoliciesByID(): void
    {
        $mockPolicies = [
            ['id' => 'policy1', 'displayName' => 'Policy 1'],
            ['id' => 'policy2', 'displayName' => 'Policy 2']
        ];

        $this->apiClientMock
            ->expects($this->once())
            ->method('getAllPages')
            ->willReturn($mockPolicies);

        $result = $this->msGraph->getConditionalAccessPoliciesByID();

        $this->assertIsArray($result);
        $this->assertArrayHasKey('policy1', $result);
        $this->assertArrayHasKey('policy2', $result);
        $this->assertEquals('Policy 1', $result['policy1']['displayName']);
    }

    public function testCachingMechanismHit(): void
    {
        // Pre-populate cache
        $_SESSION['msgraph_cache']['test_key'] = [
            'data' => ['cached' => 'data'],
            'timestamp' => time() - 100 // 100 seconds ago, within TTL
        ];

        $reflection = new \ReflectionClass($this->msGraph);
        $getCachedDataMethod = $reflection->getMethod('getCachedData');
        $getCachedDataMethod->setAccessible(true);

        $this->loggerMock
            ->expects($this->once())
            ->method('debug')
            ->with($this->stringContains('Cache hit for test_key'));

        $callbackCalled = false;
        $callback = function() use (&$callbackCalled) {
            $callbackCalled = true;
            return ['should' => 'not be called'];
        };

        $result = $getCachedDataMethod->invoke(
            $this->msGraph,
            'test_key',
            $callback,
            300
        );

        $this->assertEquals(['cached' => 'data'], $result);
        $this->assertFalse($callbackCalled);
    }

    public function testCachingMechanismMiss(): void
    {
        $this->loggerMock
            ->expects($this->once())
            ->method('debug')
            ->with('Cache miss for test_key');

        $this->loggerMock
            ->expects($this->once())
            ->method('info')
            ->with('Loading fresh data for test_key');

        $reflection = new \ReflectionClass($this->msGraph);
        $getCachedDataMethod = $reflection->getMethod('getCachedData');
        $getCachedDataMethod->setAccessible(true);

        $freshData = ['fresh' => 'data'];
        $callback = function() use ($freshData) {
            return $freshData;
        };

        $result = $getCachedDataMethod->invoke(
            $this->msGraph,
            'test_key',
            $callback,
            300
        );

        $this->assertEquals($freshData, $result);
        $this->assertArrayHasKey('test_key', $_SESSION['msgraph_cache']);
        $this->assertEquals($freshData, $_SESSION['msgraph_cache']['test_key']['data']);
    }

    public function testCachingMechanismExpired(): void
    {
        // Pre-populate with expired cache
        $_SESSION['msgraph_cache']['test_key'] = [
            'data' => ['old' => 'data'],
            'timestamp' => time() - 400 // 400 seconds ago, beyond 300s TTL
        ];

        $this->loggerMock
            ->expects($this->once())
            ->method('debug')
            ->with($this->stringContains('Cache expired for test_key'));

        $reflection = new \ReflectionClass($this->msGraph);
        $getCachedDataMethod = $reflection->getMethod('getCachedData');
        $getCachedDataMethod->setAccessible(true);

        $freshData = ['fresh' => 'data'];
        $callback = function() use ($freshData) {
            return $freshData;
        };

        $result = $getCachedDataMethod->invoke(
            $this->msGraph,
            'test_key',
            $callback,
            300
        );

        $this->assertEquals($freshData, $result);
    }

    public function testClearAllCache(): void
    {
        // Pre-populate cache
        $_SESSION['msgraph_cache'] = [
            'key1' => ['data' => 'test1', 'timestamp' => time()],
            'key2' => ['data' => 'test2', 'timestamp' => time()]
        ];

        $this->loggerMock
            ->expects($this->once())
            ->method('info')
            ->with('Cleared all MSGraph cache');

        $this->msGraph->clearAllCache();

        $this->assertEmpty($_SESSION['msgraph_cache']);
    }

    public function testClearSpecificCache(): void
    {
        // Pre-populate cache
        $_SESSION['msgraph_cache'] = [
            'key1' => ['data' => 'test1', 'timestamp' => time()],
            'key2' => ['data' => 'test2', 'timestamp' => time()]
        ];

        $this->loggerMock
            ->expects($this->once())
            ->method('debug')
            ->with('Cleared cache for key1');

        $reflection = new \ReflectionClass($this->msGraph);
        $clearCacheMethod = $reflection->getMethod('clearCache');
        $clearCacheMethod->setAccessible(true);

        $clearCacheMethod->invoke($this->msGraph, 'key1');

        $this->assertArrayNotHasKey('key1', $_SESSION['msgraph_cache']);
        $this->assertArrayHasKey('key2', $_SESSION['msgraph_cache']);
    }

    public function testArrayAccessMethods(): void
    {
        // Test that ArrayAccess methods exist (they're currently not implemented)
        $this->assertTrue(method_exists($this->msGraph, 'offsetExists'));
        $this->assertTrue(method_exists($this->msGraph, 'offsetGet'));
        $this->assertTrue(method_exists($this->msGraph, 'offsetSet'));
        $this->assertTrue(method_exists($this->msGraph, 'offsetUnset'));
    }

    /**
     * Test that the cache TTL configuration can be properly accessed
     */
    public function testCacheTTLConfiguration(): void
    {
        // Test that we can verify the cache TTL settings without relying on constructor timing
        $reflection = new \ReflectionClass($this->msGraph);
        $cacheTTLsProperty = $reflection->getProperty('cacheTTLs');
        $cacheTTLsProperty->setAccessible(true);
        $cacheTTLs = $cacheTTLsProperty->getValue($this->msGraph);

        // Verify default values exist
        $this->assertIsArray($cacheTTLs);
        $this->assertArrayHasKey('users', $cacheTTLs);
        $this->assertArrayHasKey('policies', $cacheTTLs);
        $this->assertArrayHasKey('namedLocations', $cacheTTLs);

        // Verify they are reasonable values
        $this->assertGreaterThan(0, $cacheTTLs['users']);
        $this->assertGreaterThan(0, $cacheTTLs['policies']);
        $this->assertGreaterThan(0, $cacheTTLs['namedLocations']);
    }

    /**
     * Test session initialization
     */
    public function testSessionInitialization(): void
    {
        // Verify that the constructor properly initializes the session cache
        $this->assertArrayHasKey('msgraph_cache', $_SESSION);
        $this->assertIsArray($_SESSION['msgraph_cache']);
    }
}