<?php
namespace AmgGroup\Tests;

use AmgGroup\MSUsers;
use AmgGroup\Config;
use AmgGroup\Logger;
use AmgGroup\ApiClient;
use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\MockObject;

class MSUsersTest extends TestCase
{
    private MSUsers $msUsers;
    private MockObject $configMock;
    private MockObject $loggerMock;
    private MockObject $apiClientMock;

    protected function setUp(): void
    {
        // Clear any existing session data
        if (session_status() !== PHP_SESSION_NONE) {
            session_destroy();
        }

        // Start a new session for testing
        session_start();
        $_SESSION = [];

        // Reset the MSUsers singleton instance
        $this->resetSingleton(MSUsers::class);
        $this->resetSingleton(Config::class);
        $this->resetSingleton(Logger::class);
        $this->resetSingleton(ApiClient::class);

        // Create the MSUsers instance first
        $this->msUsers = MSUsers::getInstance();

        // Now mock the dependencies using reflection
        $this->injectMockedDependencies();
    }

    protected function tearDown(): void
    {
        if (session_status() !== PHP_SESSION_NONE) {
            session_destroy();
        }

        // Reset singletons
        $this->resetSingleton(MSUsers::class);
        $this->resetSingleton(Config::class);
        $this->resetSingleton(Logger::class);
        $this->resetSingleton(ApiClient::class);
    }

    private function resetSingleton(string $className): void
    {
        $reflection = new \ReflectionClass($className);
        $instanceProperty = $reflection->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue(null);
    }

    private function injectMockedDependencies(): void
    {
        $reflection = new \ReflectionClass($this->msUsers);

        // Mock and inject Config
        $this->configMock = $this->createMock(Config::class);
        $configProperty = $reflection->getProperty('config');
        $configProperty->setAccessible(true);
        $configProperty->setValue($this->msUsers, $this->configMock);

        // Mock and inject Logger
        $this->loggerMock = $this->createMock(Logger::class);
        $loggerProperty = $reflection->getProperty('logger');
        $loggerProperty->setAccessible(true);
        $loggerProperty->setValue($this->msUsers, $this->loggerMock);

        // We don't inject ApiClient here because it's created in the getUsers method
        // We'll mock it in the specific tests that need it
    }

    public function testGetInstanceReturnsSameInstance(): void
    {
        $instance1 = MSUsers::getInstance();
        $instance2 = MSUsers::getInstance();

        $this->assertSame($instance1, $instance2);
        $this->assertInstanceOf(MSUsers::class, $instance1);
    }

    public function testOffsetExistsReturnsTrueForCachedItem(): void
    {
        // Set up the session with a cached item
        $_SESSION[MSUsers::CACHE_KEY]['users'] = [
            'data' => ['user1' => ['id' => 'user1']],
            'timestamp' => time() // Current time, so it's not expired
        ];

        $this->assertTrue($this->msUsers->offsetExists('users'));
    }

    public function testOffsetExistsReturnsFalseForNonExistentItem(): void
    {
        $this->assertFalse($this->msUsers->offsetExists('nonexistent'));
    }

    public function testOffsetExistsReturnsFalseForExpiredItem(): void
    {
        // Set up the session with an expired item
        $_SESSION[MSUsers::CACHE_KEY]['users'] = [
            'data' => ['user1' => ['id' => 'user1']],
            'timestamp' => time() - 400 // 400 seconds ago, beyond default TTL of 300
        ];

        $this->assertFalse($this->msUsers->offsetExists('users'));
    }

    public function testOffsetGetReturnsCachedItem(): void
    {
        $cachedData = ['user1' => ['id' => 'user1']];

        // Set up the session with a cached item
        $_SESSION[MSUsers::CACHE_KEY]['users'] = [
            'data' => $cachedData,
            'timestamp' => time() // Current time, so it's not expired
        ];

        $result = $this->msUsers->offsetGet('users');
        $this->assertEquals($cachedData, $result);
    }

    public function testOffsetGetFetchesFromApiWhenNotCached(): void
    {
        // Mock the ApiClient for this test
        $apiClientMock = $this->createMock(ApiClient::class);
        $apiClientMock->method('getAllPages')
            ->willReturn([
                ['id' => 'user1', 'name' => 'User One'],
                ['id' => 'user2', 'name' => 'User Two']
            ]);

        // Use reflection to replace the getInstance method of ApiClient
        $apiClientReflection = new \ReflectionClass(ApiClient::class);
        $instanceProperty = $apiClientReflection->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue($apiClientMock);

        // Call offsetGet which should fetch from API
        $result = $this->msUsers->offsetGet('users');

        // Verify the result
        $this->assertIsArray($result);
        $this->assertArrayHasKey('user1', $result);
        $this->assertArrayHasKey('user2', $result);

        // Verify it was cached
        $this->assertArrayHasKey('users', $_SESSION[MSUsers::CACHE_KEY]);
        $this->assertEquals($result, $_SESSION[MSUsers::CACHE_KEY]['users']['data']);
    }

    public function testOffsetSetStoresInSession(): void
    {
        $data = ['user1' => ['id' => 'user1']];
        $this->msUsers->offsetSet('users', $data);

        $this->assertArrayHasKey('users', $_SESSION[MSUsers::CACHE_KEY]);
        $this->assertEquals($data, $_SESSION[MSUsers::CACHE_KEY]['users']['data']);
    }

    public function testOffsetUnsetRemovesFromSession(): void
    {
        // Set up the session with a cached item
        $_SESSION[MSUsers::CACHE_KEY]['users'] = [
            'data' => ['user1' => ['id' => 'user1']],
            'timestamp' => time()
        ];

        $this->msUsers->offsetUnset('users');
        $this->assertArrayNotHasKey('users', $_SESSION[MSUsers::CACHE_KEY]);
    }

    public function testGetUsersReturnsFormattedUserList(): void
    {
        // Mock the ApiClient for this test
        $apiClientMock = $this->createMock(ApiClient::class);
        $apiClientMock->method('getAllPages')
            ->with('https://graph.microsoft.com/v1.0/users')
            ->willReturn([
                ['id' => 'user1', 'name' => 'User One'],
                ['id' => 'user2', 'name' => 'User Two']
            ]);

        // Use reflection to replace the getInstance method of ApiClient
        $apiClientReflection = new \ReflectionClass(ApiClient::class);
        $instanceProperty = $apiClientReflection->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue($apiClientMock);

        // Use reflection to call the private getUsers method
        $reflection = new \ReflectionClass($this->msUsers);
        $getUsersMethod = $reflection->getMethod('getUsers');
        $getUsersMethod->setAccessible(true);
        $result = $getUsersMethod->invoke($this->msUsers);

        // Verify the result
        $this->assertIsArray($result);
        $this->assertCount(2, $result);
        $this->assertArrayHasKey('user1', $result);
        $this->assertArrayHasKey('user2', $result);
        $this->assertEquals('User One', $result['user1']['name']);
        $this->assertEquals('User Two', $result['user2']['name']);
    }

    public function testFetchFromApiCallsCorrectMethod(): void
    {
        // Mock the ApiClient for this test
        $apiClientMock = $this->createMock(ApiClient::class);
        $apiClientMock->method('getAllPages')
            ->with('https://graph.microsoft.com/v1.0/users')
            ->willReturn([
                ['id' => 'user1', 'name' => 'User One'],
                ['id' => 'user2', 'name' => 'User Two']
            ]);

        // Use reflection to replace the getInstance method of ApiClient
        $apiClientReflection = new \ReflectionClass(ApiClient::class);
        $instanceProperty = $apiClientReflection->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue($apiClientMock);

        // Use reflection to call the private fetchFromApi method
        $reflection = new \ReflectionClass($this->msUsers);
        $fetchFromApiMethod = $reflection->getMethod('fetchFromApi');
        $fetchFromApiMethod->setAccessible(true);
        $result = $fetchFromApiMethod->invoke($this->msUsers, 'users');

        // Verify the result contains the expected data structure
        $this->assertIsArray($result);
        $this->assertArrayHasKey('user1', $result);
        $this->assertArrayHasKey('user2', $result);
    }

    public function testGetFromSessionReturnsNullForExpiredCache(): void
    {
        // Set up the session with an expired item
        $_SESSION[MSUsers::CACHE_KEY]['users'] = [
            'data' => ['user1' => ['id' => 'user1']],
            'timestamp' => time() - 400 // 400 seconds ago, beyond default TTL of 300
        ];

        // Use reflection to call the private getFromSession method
        $reflection = new \ReflectionClass($this->msUsers);
        $getFromSessionMethod = $reflection->getMethod('getFromSession');
        $getFromSessionMethod->setAccessible(true);
        $result = $getFromSessionMethod->invoke($this->msUsers, 'users');

        // Verify the result is null for expired cache
        $this->assertNull($result);

        // Verify the expired item was removed from session
        $this->assertArrayNotHasKey('users', $_SESSION[MSUsers::CACHE_KEY]);
    }

    public function testStoreInSessionSetsTimestamp(): void
    {
        $data = ['test' => 'data'];

        // Use reflection to call the private storeInSession method
        $reflection = new \ReflectionClass($this->msUsers);
        $storeInSessionMethod = $reflection->getMethod('storeInSession');
        $storeInSessionMethod->setAccessible(true);
        $storeInSessionMethod->invoke($this->msUsers, 'test_key', $data);

        // Verify the data was stored with a timestamp
        $this->assertArrayHasKey('test_key', $_SESSION[MSUsers::CACHE_KEY]);
        $this->assertEquals($data, $_SESSION[MSUsers::CACHE_KEY]['test_key']['data']);
        $this->assertArrayHasKey('timestamp', $_SESSION[MSUsers::CACHE_KEY]['test_key']);
        $this->assertIsInt($_SESSION[MSUsers::CACHE_KEY]['test_key']['timestamp']);
    }
}
