<?php

namespace AmgGroup;

class MSOAuth2
{
    /**
     * @var string $client_id Client/Application ID for Azure App Registration
     */
    private string $client_id = '';
    /**
     * @var string $client_secret Client Secret for Azure App Registration
     */
    private string $client_secret = '';
    /**
     * @var string $tenant_id Tenant ID for Azure App Registration
     */
    private string $tenant_id = 'common';
    /**
     * @var string $redirect_uri Redirect URI for Azure App Registration
     */
    private string $redirect_uri = '';
    /**
     * @var array $scopes Scopes for Azure Authentication
     */
    private array $scopes = ['https://graph.microsoft.com/User.Read'];

    /**
     * MSOAuth2 constructor.
     * Supports both DI (passing values) and non-DI (can be set via setters if needed, 
     * or passed as an associative array).
     * 
     * @param array|string $config Either an array of configuration or the client_id
     * @param string|null $client_secret
     * @param string|null $redirect_uri
     * @param string|null $tenant_id
     */
    public function __construct($config = [], ?string $client_secret = null, ?string $redirect_uri = null, ?string $tenant_id = null)
    {
        if (is_array($config)) {
            $this->client_id = $config['client_id'] ?? '';
            $this->client_secret = $config['client_secret'] ?? '';
            $this->redirect_uri = $config['redirect_uri'] ?? '';
            $this->tenant_id = $config['tenant_id'] ?? 'common';
            if (isset($config['scopes'])) {
                $this->scopes = is_array($config['scopes']) ? $config['scopes'] : explode(' ', $config['scopes']);
            }
        } else {
            $this->client_id = (string)$config;
            $this->client_secret = $client_secret ?? '';
            $this->redirect_uri = $redirect_uri ?? '';
            $this->tenant_id = $tenant_id ?? 'common';
        }
    }

    public function setClientId(string $client_id): self { $this->client_id = $client_id; return $this; }
    public function setClientSecret(string $client_secret): self { $this->client_secret = $client_secret; return $this; }
    public function setRedirectUri(string $redirect_uri): self { $this->redirect_uri = $redirect_uri; return $this; }
    public function setTenantId(string $tenant_id): self { $this->tenant_id = $tenant_id; return $this; }
    public function setScopes(array $scopes): self { $this->scopes = $scopes; return $this; }

    /**
     * Get the Login URL to redirect the user to Azure
     * 
     * @param string|null $state Optional state for CSRF protection
     * @return string
     */
    public function getLoginUrl(?string $state = null): string
    {
        $params = [
            'client_id' => $this->client_id,
            'redirect_uri' => $this->redirect_uri,
            'response_type' => 'token',
            'response_mode' => 'form_post',
            'scope' => implode(' ', $this->scopes),
            'state' => $state
        ];

        return "https://login.microsoftonline.com/" . $this->tenant_id . "/oauth2/v2.0/authorize?" . http_build_query($params);
    }

    /**
     * Handle the callback from Azure. 
     * In the sample code, this expects an 'access_token' in $_POST.
     * 
     * @param array $postData Usually $_POST
     * @return string|null The access token if successful, null otherwise
     */
    public function handleCallback(array $postData): ?string
    {
        return $postData['access_token'] ?? null;
    }

    /**
     * Get user details from Microsoft Graph API
     * 
     * @param string $accessToken
     * @return array|null
     */
    public function getUserDetails(string $accessToken): ?array
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_URL, "https://graph.microsoft.com/v1.0/me/");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        $result = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return ['error' => $error];
        }

        return json_decode($result, true);
    }

    /**
     * Get the Logout URL to redirect the user to Azure for logout
     * 
     * @param string|null $postLogoutRedirectUri Optional redirect URI after logout
     * @return string
     */
    public function getLogoutUrl(?string $postLogoutRedirectUri = null): string
    {
        $url = "https://login.microsoftonline.com/" . $this->tenant_id . "/oauth2/v2.0/logout";
        if ($postLogoutRedirectUri || $this->redirect_uri) {
            $url .= "?" . http_build_query(['post_logout_redirect_uri' => $postLogoutRedirectUri ?? $this->redirect_uri]);
        }
        return $url;
    }

    /**
     * Returns a basic HTML login button
     * 
     * @param string $label
     * @param string $state
     * @param array $attributes Optional HTML attributes
     * @return string
     */
    public function getLoginButton(string $label = 'Log In with Microsoft', ?string $state = null, array $attributes = []): string
    {
        $url = $this->getLoginUrl($state);
        $attrString = '';
        foreach ($attributes as $key => $value) {
            $attrString .= ' ' . htmlspecialchars($key) . '="' . htmlspecialchars($value) . '"';
        }
        return '<a href="' . htmlspecialchars($url) . '"' . $attrString . '>' . htmlspecialchars($label) . '</a>';
    }

    /**
     * Returns a basic HTML logout button
     * 
     * @param string $label
     * @param string|null $postLogoutRedirectUri
     * @param array $attributes Optional HTML attributes
     * @return string
     */
    public function getLogoutButton(string $label = 'Log Out', ?string $postLogoutRedirectUri = null, array $attributes = []): string
    {
        $url = $this->getLogoutUrl($postLogoutRedirectUri);
        $attrString = '';
        foreach ($attributes as $key => $value) {
            $attrString .= ' ' . htmlspecialchars($key) . '="' . htmlspecialchars($value) . '"';
        }
        return '<a href="' . htmlspecialchars($url) . '"' . $attrString . '>' . htmlspecialchars($label) . '</a>';
    }

    /**
     * Returns login data suitable for templates
     * 
     * @param string|null $state
     * @return array
     */
    public function getLoginData(?string $state = null): array
    {
        return [
            'url' => $this->getLoginUrl($state),
            'label' => 'Log In with Microsoft'
        ];
    }

    /**
     * Returns logout data suitable for templates
     * 
     * @param string|null $postLogoutRedirectUri
     * @return array
     */
    public function getLogoutData(?string $postLogoutRedirectUri = null): array
    {
        return [
            'url' => $this->getLogoutUrl($postLogoutRedirectUri),
            'label' => 'Log Out'
        ];
    }

    /**
     * Check if the user is logged in.
     * Based on the sample code, this checks for the presence of 'msatg' in the session.
     * 
     * @return bool
     */
    public function isLoggedIn(): bool
    {
        if (session_status() === PHP_SESSION_NONE && !headers_sent()) {
            session_start();
        }
        return isset($_SESSION['msatg']) && $_SESSION['msatg'] == 1;
    }
}