# MSOAuth2 - Microsoft Azure SSO Authentication Class

A lean, dependency-free PHP class for integrating Microsoft Azure SSO (OAuth 2.0) into your applications. It supports both Dependency Injection (DI) and standard instantiation, making it easy to use in various project architectures.

## Features

- **Lean Implementation**: Uses standard PHP and cURL (no external dependencies).
- **DI & Non-DI Support**: Flexible constructor and setters for all configurations.
- **Template Friendly**: Methods to return raw data for Smarty or other template engines.
- **HTML Helpers**: Built-in methods to generate login/logout buttons.
- **Session Integration**: Easy login status checking.

## Configuration Parameters

- `client_id`: The Application (client) ID from your Azure App Registration.
- `client_secret`: The Client Secret from your Azure App Registration.
- `redirect_uri`: The URI where Azure will send the user after authentication.
- `tenant_id`: The Directory (tenant) ID (defaults to 'common').
- `scopes`: An array of permissions (defaults to `['https://graph.microsoft.com/User.Read']`).

---

## Usage Examples

### 1. Basic (Non-DI) Usage

If you are not using a DI container, you can pass an associative array to the constructor.

```php
require_once 'vendor/autoload.php';

use AmgGroup\MSOAuth2;

$config = [
    'client_id' => 'your-client-id',
    'client_secret' => 'your-client-secret',
    'redirect_uri' => 'https://your-app.com/callback.php',
    'tenant_id' => 'common'
];

$auth = new MSOAuth2($config);

if (!$auth->isLoggedIn()) {
    echo $auth->getLoginButton("Login with Microsoft", null, ['class' => 'btn btn-primary']);
} else {
    echo "Welcome back!";
    echo $auth->getLogoutButton("Logout", "https://your-app.com/home.php");
}
```

### 2. Dependency Injection (DI) Usage

The class is designed to be easily injected into your services or controllers.

```php
use AmgGroup\MSOAuth2;

// In your DI Container configuration
$container->set(MSOAuth2::class, function() {
    return new MSOAuth2(
        'your-client-id',
        'your-client-secret',
        'https://your-app.com/callback.php',
        'your-tenant-id'
    );
});

// In your Controller
class LoginController {
    public function __construct(private MSOAuth2 $auth) {}

    public function index() {
        if ($this->auth->isLoggedIn()) {
            // Redirect to dashboard
        }
        return $this->render('login', ['loginUrl' => $this->auth->getLoginUrl()]);
    }
}
```

### 3. Handling the Callback (Router)

In your callback file or route:

```php
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['access_token'])) {
    $token = $auth->handleCallback($_POST);
    
    if ($token) {
        $userDetails = $auth->getUserDetails($token);
        
        if ($userDetails && !isset($userDetails['error'])) {
            // Success! Store user info in session
            $_SESSION['msatg'] = 1;
            $_SESSION['uname'] = $userDetails['displayName'];
            $_SESSION['id'] = $userDetails['id'];
            
            header('Location: /dashboard');
            exit;
        }
    }
}
```

### 4. Smarty Template Integration

If you want to pass data to a template instead of generating HTML in PHP:

```php
// In PHP
$smarty->assign('msLogin', $auth->getLoginData());

// In Smarty Template (login.tpl)
<a href="{$msLogin.url}" class="ms-btn">{$msLogin.label}</a>
```

---

## Method Reference

### `__construct($config = [], ?string $client_secret = null, ?string $redirect_uri = null, ?string $tenant_id = null)`
Initializes the class. `$config` can be an associative array of all settings or just the `client_id` string.

### `isLoggedIn(): bool`
Checks if the user is currently logged in (looks for `$_SESSION['msatg'] == 1`). Automatically starts the session if not already started.

### `getLoginUrl(?string $state = null): string`
Generates the authorization URL for Microsoft Azure. Use `$state` for CSRF protection.

### `getLoginButton(string $label, ?string $state, array $attributes): string`
Returns a full HTML `<a>` tag for the login button. `$attributes` is an associative array of HTML attributes (e.g., `['class' => 'my-css-class']`).

### `getLoginData(?string $state = null): array`
Returns an array `['url' => ..., 'label' => ...]` for use in templates.

### `handleCallback(array $postData): ?string`
Extracts the `access_token` from the POST data returned by Azure.

### `getUserDetails(string $accessToken): ?array`
Fetches user profile data from the Microsoft Graph API using the provided access token.

### `getLogoutUrl(?string $postLogoutRedirectUri = null): string`
Generates the logout URL. Defaults to the configured `redirect_uri` if no parameter is provided.

### `getLogoutButton(string $label, ?string $postLogoutRedirectUri, array $attributes): string`
Returns a full HTML `<a>` tag for the logout button.

### `getLogoutData(?string $postLogoutRedirectUri = null): array`
Returns an array `['url' => ..., 'label' => ...]` for logout, suitable for templates.

---

## Setters
For fine-grained control, you can also use:
- `setClientId(string $id)`
- `setClientSecret(string $secret)`
- `setRedirectUri(string $uri)`
- `setTenantId(string $id)`
- `setScopes(array $scopes)`

---

## Running Tests

This project uses PHPUnit for testing. You can run the tests using Composer:

```bash
composer test
```

Or run PHPUnit directly:

```bash
./vendor/bin/phpunit
```

The tests cover:
- Class instantiation (DI and Non-DI)
- URL generation for Login and Logout
- Callback handling and token extraction
- HTML button generation
- Template data structure
- Session-based login status checking
- Error handling for API calls (cURL)
