<?php

namespace AmgGroup\Tests;

use AmgGroup\Logger;
use AmgGroup\SpecializedLogger;
use AmgGroup\Tests\Mocks\Config;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

/**
 * Test case for the SpecializedLogger class
 */
class SpecializedLoggerTest extends TestCase
{
    /**
     * Set up before each test
     */
    protected function setUp(): void
    {
        // Reset the Logger instance before each test
        $reflectionClass = new ReflectionClass(Logger::class);
        $instanceProperty = $reflectionClass->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue(null, null);

        // Reset the specialized loggers
        $specializedLoggersProperty = $reflectionClass->getProperty('specializedLoggers');
        $specializedLoggersProperty->setAccessible(true);
        $specializedLoggersProperty->setValue(null, []);

        // Reset the Config instance
        Config::resetInstance();
    }

    /**
     * Test creating a specialized logger
     */
    public function testCreateSpecializedLogger(): void
    {
        $context = 'test-context';
        $filePath = 'test-specialized.log';

        $specializedLogger = Logger::getSpecializedLogger($context, $filePath);

        $this->assertInstanceOf(SpecializedLogger::class, $specializedLogger);
    }

    /**
     * Test that specialized logger logs to both loggers
     */
    public function testSpecializedLoggerLogsToBothLoggers(): void
    {
        // Create a mock main logger
        $mainLogger = $this->createMock(Logger::class);

        // Create a mock file logger
        $fileLogger = $this->createMock(\Monolog\Logger::class);

        // Create a specialized logger with reflection to inject our mocks
        $specializedLogger = new SpecializedLogger($mainLogger, 'test-context', 'test.log', Logger::INFO);

        // Use reflection to replace the file logger
        $reflectionClass = new ReflectionClass(SpecializedLogger::class);
        $fileLoggerProperty = $reflectionClass->getProperty('fileLogger');
        $fileLoggerProperty->setAccessible(true);
        $fileLoggerProperty->setValue($specializedLogger, $fileLogger);

        // Set expectations for the debug method
        $mainLogger->expects($this->once())
            ->method('debug')
            ->with($this->stringContains('[test-context]'), $this->equalTo(['test' => 'data']));

        $fileLogger->expects($this->once())
            ->method('debug')
            ->with('Test debug message', $this->equalTo(['test' => 'data']));

        // Call the debug method
        $specializedLogger->debug('Test debug message', ['test' => 'data']);
    }

    /**
     * Test that specialized logger logs at info level
     */
    public function testSpecializedLoggerLogsAtInfoLevel(): void
    {
        // Create a mock main logger
        $mainLogger = $this->createMock(Logger::class);

        // Create a mock file logger
        $fileLogger = $this->createMock(\Monolog\Logger::class);

        // Create a specialized logger with reflection to inject our mocks
        $specializedLogger = new SpecializedLogger($mainLogger, 'test-context', 'test.log', Logger::INFO);

        // Use reflection to replace the file logger
        $reflectionClass = new ReflectionClass(SpecializedLogger::class);
        $fileLoggerProperty = $reflectionClass->getProperty('fileLogger');
        $fileLoggerProperty->setAccessible(true);
        $fileLoggerProperty->setValue($specializedLogger, $fileLogger);

        // Set expectations for the info method
        $mainLogger->expects($this->once())
            ->method('info')
            ->with($this->stringContains('[test-context]'), $this->equalTo(['test' => 'data']));

        $fileLogger->expects($this->once())
            ->method('info')
            ->with('Test info message', $this->equalTo(['test' => 'data']));

        // Call the info method
        $specializedLogger->info('Test info message', ['test' => 'data']);
    }

    /**
     * Test that specialized logger logs at warning level
     */
    public function testSpecializedLoggerLogsAtWarningLevel(): void
    {
        // Create a mock main logger
        $mainLogger = $this->createMock(Logger::class);

        // Create a mock file logger
        $fileLogger = $this->createMock(\Monolog\Logger::class);

        // Create a specialized logger with reflection to inject our mocks
        $specializedLogger = new SpecializedLogger($mainLogger, 'test-context', 'test.log', Logger::INFO);

        // Use reflection to replace the file logger
        $reflectionClass = new ReflectionClass(SpecializedLogger::class);
        $fileLoggerProperty = $reflectionClass->getProperty('fileLogger');
        $fileLoggerProperty->setAccessible(true);
        $fileLoggerProperty->setValue($specializedLogger, $fileLogger);

        // Set expectations for the warning method
        $mainLogger->expects($this->once())
            ->method('warning')
            ->with($this->stringContains('[test-context]'), $this->equalTo(['test' => 'data']));

        $fileLogger->expects($this->once())
            ->method('warning')
            ->with('Test warning message', $this->equalTo(['test' => 'data']));

        // Call the warning method
        $specializedLogger->warning('Test warning message', ['test' => 'data']);
    }

    /**
     * Test that specialized logger logs at error level
     */
    public function testSpecializedLoggerLogsAtErrorLevel(): void
    {
        // Create a mock main logger
        $mainLogger = $this->createMock(Logger::class);

        // Create a mock file logger
        $fileLogger = $this->createMock(\Monolog\Logger::class);

        // Create a specialized logger with reflection to inject our mocks
        $specializedLogger = new SpecializedLogger($mainLogger, 'test-context', 'test.log', Logger::INFO);

        // Use reflection to replace the file logger
        $reflectionClass = new ReflectionClass(SpecializedLogger::class);
        $fileLoggerProperty = $reflectionClass->getProperty('fileLogger');
        $fileLoggerProperty->setAccessible(true);
        $fileLoggerProperty->setValue($specializedLogger, $fileLogger);

        // Set expectations for the error method
        $mainLogger->expects($this->once())
            ->method('error')
            ->with($this->stringContains('[test-context]'), $this->equalTo(['test' => 'data']));

        $fileLogger->expects($this->once())
            ->method('error')
            ->with('Test error message', $this->equalTo(['test' => 'data']));

        // Call the error method
        $specializedLogger->error('Test error message', ['test' => 'data']);
    }

    /**
     * Test that specialized logger logs with generic log method
     */
    public function testSpecializedLoggerLogsWithGenericLogMethod(): void
    {
        // Create a mock main logger
        $mainLogger = $this->createMock(Logger::class);

        // Create a mock file logger
        $fileLogger = $this->createMock(\Monolog\Logger::class);

        // Create a specialized logger with reflection to inject our mocks
        $specializedLogger = new SpecializedLogger($mainLogger, 'test-context', 'test.log', Logger::INFO);

        // Use reflection to replace the file logger
        $reflectionClass = new ReflectionClass(SpecializedLogger::class);
        $fileLoggerProperty = $reflectionClass->getProperty('fileLogger');
        $fileLoggerProperty->setAccessible(true);
        $fileLoggerProperty->setValue($specializedLogger, $fileLogger);

        // Set expectations for the log method
        $mainLogger->expects($this->once())
            ->method('log')
            ->with('notice', $this->stringContains('[test-context]'), $this->equalTo(['test' => 'data']));

        $fileLogger->expects($this->once())
            ->method('log')
            ->with('notice', 'Test notice message', $this->equalTo(['test' => 'data']));

        // Call the log method
        $specializedLogger->log('notice', 'Test notice message', ['test' => 'data']);
    }
    /**
     * Test that __get method returns a LogLevelProxy instance for valid log levels
     */
    public function testGetReturnsLogLevelProxyForValidLogLevels(): void
    {
        // Create a mock main logger
        $mainLogger = $this->createMock(Logger::class);

        // Create a specialized logger
        $specializedLogger = new SpecializedLogger($mainLogger, 'test-context', 'test.log', Logger::INFO);

        // Test all valid log levels
        $validLevels = ['debug', 'info', 'notice', 'warning', 'error', 'critical', 'alert', 'emergency'];

        foreach ($validLevels as $level) {
            $proxy = $specializedLogger->$level;
            $this->assertInstanceOf('AmgGroup\LogLevelProxy', $proxy);

            // Test that the proxy is cached (same instance returned for same level)
            $proxy2 = $specializedLogger->$level;
            $this->assertSame($proxy, $proxy2);
        }
    }

    /**
     * Test that __get method throws an exception for invalid log levels
     */
    public function testGetThrowsExceptionForInvalidLogLevels(): void
    {
        // Create a mock main logger
        $mainLogger = $this->createMock(Logger::class);

        // Create a specialized logger
        $specializedLogger = new SpecializedLogger($mainLogger, 'test-context', 'test.log', Logger::INFO);

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('Invalid log level: invalid');

        // This should throw an exception
        $specializedLogger->invalid;
    }
}
