<?php

namespace AmgGroup\Tests;

use AmgGroup\Logger;
use AmgGroup\Tests\Mocks\Config;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

/**
 * Test case for the Logger class
 */
class LoggerTest extends TestCase
{
    /**
     * Set up before each test
     */
    protected function setUp(): void
    {
        // Reset the Logger instance before each test
        $reflectionClass = new ReflectionClass(Logger::class);
        $instanceProperty = $reflectionClass->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue(null, null);

        // Reset the specialized loggers
        $specializedLoggersProperty = $reflectionClass->getProperty('specializedLoggers');
        $specializedLoggersProperty->setAccessible(true);
        $specializedLoggersProperty->setValue(null, []);

        // Reset the Config instance
        Config::resetInstance();
    }

    /**
     * Test that getInstance returns a singleton instance
     */
    public function testGetInstance(): void
    {
        // First call should create a new instance
        $logger1 = Logger::getInstance();
        $this->assertInstanceOf(Logger::class, $logger1);

        // Second call should return the same instance
        $logger2 = Logger::getInstance();
        $this->assertSame($logger1, $logger2);
    }

    /**
     * Test extended tracing functionality
     */
    public function testExtendedTracing(): void
    {
        $logger = Logger::getInstance();

        // Default should be false
        $this->assertFalse($logger->isExtendedTracingEnabled());

        // Enable extended tracing
        $result = $logger->extended(true);
        $this->assertTrue($logger->isExtendedTracingEnabled());
        $this->assertSame($logger, $result, 'Method should return $this for chaining');

        // Disable extended tracing
        $result = $logger->extended(false);
        $this->assertFalse($logger->isExtendedTracingEnabled());
        $this->assertSame($logger, $result, 'Method should return $this for chaining');
    }

    /**
     * Test setting extended tracing depth
     */
    public function testSetExtendedTracingDepth(): void
    {
        $logger = Logger::getInstance();

        // Set a valid depth
        $result = $logger->setExtendedTracingDepth(5);

        // Use reflection to check the private property
        $reflectionClass = new ReflectionClass(Logger::class);
        $depthProperty = $reflectionClass->getProperty('extendedTracingDepth');
        $depthProperty->setAccessible(true);
        $this->assertEquals(5, $depthProperty->getValue($logger));
        $this->assertSame($logger, $result, 'Method should return $this for chaining');

        // Test with a value less than 1 (should be set to 1)
        $logger->setExtendedTracingDepth(0);
        $this->assertEquals(1, $depthProperty->getValue($logger), 'Depth should be at least 1');
    }

    /**
     * Test getting a specialized logger
     */
    public function testGetSpecializedLogger(): void
    {
        // Get a specialized logger
        $specializedLogger = Logger::getSpecializedLogger('test-context');

        // Check that it's the right type
        $this->assertInstanceOf('AmgGroup\SpecializedLogger', $specializedLogger);

        // Getting the same context again should return the same instance
        $specializedLogger2 = Logger::getSpecializedLogger('test-context');
        $this->assertSame($specializedLogger, $specializedLogger2);

        // Getting a different context should return a different instance
        $specializedLogger3 = Logger::getSpecializedLogger('other-context');
        $this->assertNotSame($specializedLogger, $specializedLogger3);
    }

    /**
     * Test specialized logger extended tracing
     */
    public function testSpecializedLoggerExtendedTracing(): void
    {
        $specializedLogger = Logger::getSpecializedLogger('test-context');

        // Default should be false
        $this->assertFalse($specializedLogger->isExtendedTracingEnabled());

        // Enable extended tracing
        $result = $specializedLogger->extended(true);
        $this->assertTrue($specializedLogger->isExtendedTracingEnabled());
        $this->assertSame($specializedLogger, $result, 'Method should return $this for chaining');

        // Disable extended tracing
        $result = $specializedLogger->extended(false);
        $this->assertFalse($specializedLogger->isExtendedTracingEnabled());
        $this->assertSame($specializedLogger, $result, 'Method should return $this for chaining');
    }

    /**
     * Test logger with configuration
     */
    public function testLoggerWithConfig(): void
    {
        // Set up config with custom values
        $config = Config::getInstance();
        $config->set('debug.enable', true);
        $config->set('debug.level', 'ERROR');
        $config->set('debug.file', 'test.log');
        $config->set('debug.console', true);

        // Get logger instance
        $logger = Logger::getInstance();

        // We can't easily test the actual logging behavior without mocking handlers,
        // but we can verify the logger was created without errors
        $this->assertInstanceOf(Logger::class, $logger);
    }
    /**
     * Test that __get method returns a LogLevelProxy instance for valid log levels
     */
    public function testGetReturnsLogLevelProxyForValidLogLevels(): void
    {
        $logger = Logger::getInstance();

        // Test all valid log levels
        $validLevels = ['debug', 'info', 'notice', 'warning', 'error', 'critical', 'alert', 'emergency'];

        foreach ($validLevels as $level) {
            $proxy = $logger->$level;
            $this->assertInstanceOf('AmgGroup\LogLevelProxy', $proxy);

            // Test that the proxy is cached (same instance returned for same level)
            $proxy2 = $logger->$level;
            $this->assertSame($proxy, $proxy2);
        }
    }

    /**
     * Test that __get method throws an exception for invalid log levels
     */
    public function testGetThrowsExceptionForInvalidLogLevels(): void
    {
        $logger = Logger::getInstance();

        $this->expectException(\InvalidArgumentException::class);
        $this->expectExceptionMessage('Invalid log level: invalid');

        // This should throw an exception
        $logger->invalid;
    }
}
