<?php

namespace AmgGroup\Tests;

use AmgGroup\Logger;
use AmgGroup\LogLevelProxy;
use AmgGroup\SpecializedLogger;
use AmgGroup\Tests\Mocks\Config;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

/**
 * Test case for the LogLevelProxy class
 */
class LogLevelProxyTest extends TestCase
{
    /**
     * Set up before each test
     */
    protected function setUp(): void
    {
        // Reset the Logger instance before each test
        $reflectionClass = new ReflectionClass(Logger::class);
        $instanceProperty = $reflectionClass->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue(null, null);
        
        // Reset the specialized loggers
        $specializedLoggersProperty = $reflectionClass->getProperty('specializedLoggers');
        $specializedLoggersProperty->setAccessible(true);
        $specializedLoggersProperty->setValue(null, []);
        
        // Reset the Config instance
        Config::resetInstance();
    }
    
    /**
     * Test that LogLevelProxy correctly captures method calls as tags
     */
    public function testLogLevelProxyCapturesMethodCallsAsTags(): void
    {
        // Create a mock logger
        $logger = $this->createMock(Logger::class);
        
        // Create a LogLevelProxy instance
        $proxy = new LogLevelProxy($logger, 'info');
        
        // Set expectations for the info method
        $logger->expects($this->once())
            ->method('info')
            ->with(
                $this->equalTo('Test message'),
                $this->equalTo(['test' => 'data', 'tag' => 'auth'])
            );
        
        // Call a method on the proxy
        $proxy->auth('Test message', ['test' => 'data']);
    }
    
    /**
     * Test that LogLevelProxy works with different log levels
     */
    public function testLogLevelProxyWorksWithDifferentLogLevels(): void
    {
        // Create a mock logger
        $logger = $this->createMock(Logger::class);
        
        // Create LogLevelProxy instances for different levels
        $debugProxy = new LogLevelProxy($logger, 'debug');
        $errorProxy = new LogLevelProxy($logger, 'error');
        
        // Set expectations for the debug method
        $logger->expects($this->once())
            ->method('debug')
            ->with(
                $this->equalTo('Debug message'),
                $this->equalTo(['tag' => 'api'])
            );
        
        // Set expectations for the error method
        $logger->expects($this->once())
            ->method('error')
            ->with(
                $this->equalTo('Error message'),
                $this->equalTo(['tag' => 'database'])
            );
        
        // Call methods on the proxies
        $debugProxy->api('Debug message');
        $errorProxy->database('Error message');
    }
    
    /**
     * Test that LogLevelProxy works with SpecializedLogger
     */
    public function testLogLevelProxyWorksWithSpecializedLogger(): void
    {
        // Create a mock specialized logger
        $specializedLogger = $this->createMock(SpecializedLogger::class);
        
        // Create a LogLevelProxy instance
        $proxy = new LogLevelProxy($specializedLogger, 'info');
        
        // Set expectations for the info method
        $specializedLogger->expects($this->once())
            ->method('info')
            ->with(
                $this->equalTo('Test message'),
                $this->equalTo(['test' => 'data', 'tag' => 'auth'])
            );
        
        // Call a method on the proxy
        $proxy->auth('Test message', ['test' => 'data']);
    }
    
    /**
     * Test that LogLevelProxy handles empty arguments correctly
     */
    public function testLogLevelProxyHandlesEmptyArgumentsCorrectly(): void
    {
        // Create a mock logger
        $logger = $this->createMock(Logger::class);
        
        // Create a LogLevelProxy instance
        $proxy = new LogLevelProxy($logger, 'info');
        
        // Set expectations for the info method with empty message and context
        $logger->expects($this->once())
            ->method('info')
            ->with(
                $this->equalTo(''),
                $this->equalTo(['tag' => 'auth'])
            );
        
        // Call a method on the proxy without arguments
        $proxy->auth();
    }
}