<?php

namespace AmgGroup;

use Monolog\Logger as MonologLogger;
use Monolog\Handler\StreamHandler;
use Monolog\Handler\ErrorLogHandler;
use Monolog\Handler\NullHandler;
use Monolog\Handler\BrowserConsoleHandler;
use Monolog\Processor\IntrospectionProcessor;
use Monolog\ErrorHandler;
use Monolog\Processor\ProcessIdProcessor;
use Monolog\Processor\MemoryUsageProcessor;
use Monolog\Formatter\JsonFormatter;
use Monolog\Formatter\FormatterInterface;

/**
 * Class Logger
 *
 * A custom logger that extends Monolog and integrates configuration settings.
 * Refactored as a Singleton to ensure a single instance is used throughout the application.
 */
class Logger extends MonologLogger
{
    /**
     * Holds the single instance of Logger.
     *
     * @var Logger|null
     */
    private static ?Logger $instance = null;

    /**
     * Store specialized loggers for different contexts
     *
     * @var array
     */
    private static array $specializedLoggers = [];

    /**
     * Flag to enable extended call stack tracing
     *
     * @var bool
     */
    private bool $extendedTracing = false;
    /**
     * How many levels of call stack to include in extended tracing
     *
     * @var int
     */
    private int $extendedTracingDepth = 2;

    /**
     * Filter rules for log entries
     *
     * @var array
     */
    private array $filterRules = [];

    /**
     * Mapping of log level names to Monolog's constants.
     */
    private const LEVEL_MAP = [
        'DEBUG' => self::DEBUG,
        'INFO' => self::INFO,
        'NOTICE' => self::NOTICE,
        'WARNING' => self::WARNING,
        'ERROR' => self::ERROR,
        'CRITICAL' => self::CRITICAL,
        'ALERT' => self::ALERT,
        'EMERGENCY' => self::EMERGENCY,
    ];

    /**
     * Private constructor to prevent direct instantiation.
     *
     * Initializes the logger with handlers based on configuration settings.
     *
     * @param Config $configLoader The configuration loader instance.
     *
     * @throws \Exception If any handler fails to initialize.
     */
    private function __construct(Config $configLoader = null)
    {
        parent::__construct('app');

        if ($configLoader === null) {
            /*
             * No config class passed, so we use a basic set of config options
             * to ensure we have some logging.
             */
            $debugEnabled = true; // Enable debugging by default when no config is provided
            $debugFile = tempnam(sys_get_temp_dir(), 'amgLogger_');
            $debugConsole = true;
            $debugSystemErrors = true;
            $debugIntrospection = true;
            $debugFatal = true;
            $debugLevelName = 'INFO';
            $this->filterRules=[
                'include' => [
                    'file' => [],
                    'line' => [],
                    'class' => [],
                    'method' => []
                ],
                'exclude' => [
                    'file' => [],
                    'line' => [],
                    'class' => [],
                    'method' => []
                ]
            ];
        } else {
            // Load configuration
            $debugEnabled = $configLoader->get('debug.enable', false);
            $debugLevelName = strtoupper($configLoader->get('debug.level', 'DEBUG')); // Default to DEBUG
            $debugFile = $configLoader->get('debug.file', null);
            $debugConsole = $configLoader->get('debug.console', false);
            $debugSystemErrors = $configLoader->get('debug.system_errors', false);
            $debugIntrospection = $configLoader->get('debug.introspection', false);
            $debugFatal = $configLoader->get('debug.fatal', false);
            // Load filter configuration
            $this->filterRules = [
                'include' => [
                    'file' => $configLoader->get('debug.include.file', []),
                    'line' => $configLoader->get('debug.include.line', []),
                    'class' => $configLoader->get('debug.include.class', []),
                    'method' => $configLoader->get('debug.include.method', [])
                ],
                'exclude' => [
                    'file' => $configLoader->get('debug.exclude.file', []),
                    'line' => $configLoader->get('debug.exclude.line', []),
                    'class' => $configLoader->get('debug.exclude.class', []),
                    'method' => $configLoader->get('debug.exclude.method', [])
                ]
            ];
        }
        // Convert level name to Monolog's log level constant
        $debugLevel = self::LEVEL_MAP[$debugLevelName] ?? self::DEBUG;

        // If debugging is disabled, use a NullHandler (no logging)
        if (!$debugEnabled) {
            $this->pushHandler(new NullHandler());
            return;
        }

        // Add a file handler if specified
        if ($debugFile) {
            //$this->pushHandler(new StreamHandler($debugFile, $debugLevel));
            $handler = new StreamHandler($debugFile, $debugLevel);
            //$formatter = new JsonFormatter(JsonFormatter::BATCH_MODE_JSON, true, true); // true enables pretty printing
            $handler->setFormatter(new PrettyArrayFormatter());;
            $this->pushHandler($handler);

        }

        // Add a console log handler if enabled
        if ($debugConsole) {
            $this->pushHandler(new BrowserConsoleHandler($debugLevel));
        }

        // By default, add an error log handler (logs to PHP error log)
        if (!$debugFile && !$debugConsole) {
            $this->pushHandler(new ErrorLogHandler(ErrorLogHandler::OPERATING_SYSTEM, $debugLevel));
        }

        if ($debugSystemErrors) {
            // Register the error handler to catch PHP errors
            $errorHandler = ErrorHandler::register($this);
            $errorHandler->registerExceptionHandler();
        }

        if ($debugFatal) {
            // If we already have an error handler instance, use it
            if (isset($errorHandler)) {
                $errorHandler->registerFatalHandler();
            } else {
                // Otherwise create a new one just for fatal errors
                $errorHandler = ErrorHandler::register($this, [], false);
                $errorHandler->registerFatalHandler();
            }
        }

        if ($debugIntrospection) {
            // Use a custom processor instead of IntrospectionProcessor
            $this->pushProcessor(function ($record) {
                // Limit trace depth to prevent excessive memory usage
                $trace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 20);

                // Find the first call outside the logging framework
                $i = 0;
                $callChain = [];

                // Skip internal logger classes
                while (isset($trace[$i])) {
                    $class = $trace[$i]['class'] ?? '';

                    if (!empty($class) && (
                            strpos($class, 'Monolog\\') === 0 ||
                            strpos($class, 'Amg\\Logger') === 0 ||
                            strpos($class, 'Amg\\SpecializedLogger') === 0
                        )) {
                        $i++;
                        continue;
                    }

                    // Found first non-logger class - this is our immediate caller
                    break;
                }

                // Store the immediate caller info (most recent in the chain)
                if (isset($trace[$i])) {
                    $record['extra']['file'] = $trace[$i]['file'] ?? '';
                    $record['extra']['line'] = $trace[$i]['line'] ?? '';
                    $record['extra']['class'] = $trace[$i]['class'] ?? '';
                    $record['extra']['function'] = $trace[$i]['function'] ?? '';
                }

                // If extended tracing is enabled, collect the call chain
                if ($this->extendedTracing) {
                    $callChain = [];

                    // Start with the immediate caller and work backward through the stack
                    // to collect all previous callers in the chain
                    $depth = 0;
                    $currentIndex = $i;

                    while (isset($trace[$currentIndex]) && $depth < $this->extendedTracingDepth) {
                        $caller = [
                            'file' => $trace[$currentIndex]['file'] ?? '',
                            'line' => $trace[$currentIndex]['line'] ?? '',
                            'class' => $trace[$currentIndex]['class'] ?? '',
                            'function' => $trace[$currentIndex]['function'] ?? ''
                        ];

                        $callChain[] = $caller;
                        $currentIndex++;
                        $depth++;
                    }

                    $record['extra']['call_chain'] = $callChain;
                }

                // Check if this record should be logged based on filter rules
                if (!$this->shouldLog($record)) {
                    // Handle both array and object records for Monolog v3 compatibility
                    if (is_object($record) && method_exists($record, 'setLevel')) {
                        $record->setLevel(Logger::DEBUG - 1);
                    } elseif (is_array($record)) {
                        $record['level'] = Logger::DEBUG - 1;
                    }
                }

                return $record;
            });

        }


        // Add stack trace processor to include debug_backtrace conditionally based on log level
        $this->pushProcessor(function ($record) {
            // Define the levels for which the stack trace should be added
            $levelsWithStackTrace = [
                Logger::ERROR,
                Logger::CRITICAL,
                Logger::ALERT,
                Logger::EMERGENCY,
            ];

            // Handle both array records and Monolog v3 LogRecord objects
            $level = is_object($record) ? $record->level->value : ($record['level'] ?? 0);

            if (in_array($level, $levelsWithStackTrace, true)) {
                if (is_object($record)) {
                    $record->extra['stack_trace'] = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 20);
                } else {
                    $record['extra']['stack_trace'] = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 20);
                }
            }

            return $record;
        });


        // Add other useful processors if needed (like process ID or memory usage)
        $this->pushProcessor(new ProcessIdProcessor());
        $this->pushProcessor(new MemoryUsageProcessor());
    }

    /**
     * Flag to prevent circular dependency with Config
     * 
     * @var bool
     */
    private static bool $isInitializing = false;

    /**
     * Provides access to the single instance of Logger.
     *
     * @return Logger
     * @throws \Exception
     */
    public static function getInstance(): Logger
    {
        // If we're already initializing, return a basic logger to break the circular dependency
        if (self::$isInitializing) {
            if (self::$instance === null) {
                self::$instance = new self(null);
            }
            return self::$instance;
        }

        try {
            self::$isInitializing = true;
            $configLoader = Config::getInstance();
            if (self::$instance === null) {
                self::$instance = new self($configLoader);
            }
        } catch (\Throwable $e) {
            // If Config class can't be loaded or getInstance fails, create Logger with null config
            if (self::$instance === null) {
                self::$instance = new self(null);
            }
        } finally {
            self::$isInitializing = false;
        }
        return self::$instance;
    }

    /**
     * Get or create a specialized logger for a specific context.
     * This logger writes to both the main log and a specialized log file.
     *
     * @param string $context The context name (used for the file name)
     * @param string|null $filePath The path to the log file (defaults to "./specialized-{context}.log")
     * @param string $levelName The log level (defaults to "DEBUG")
     * @return SpecializedLogger A specialized logger instance
     * @throws \Exception If logger creation fails
     */
    public static function getSpecializedLogger(string $context, string|null $filePath = null, string $levelName = 'DEBUG'): SpecializedLogger
    {
        // Generate a key for caching
        $key = $context . ($filePath ?? '');

        // Return existing specialized logger if it exists
        if (isset(self::$specializedLoggers[$key])) {
            return self::$specializedLoggers[$key];
        }

        // Get main logger instance
        $mainLogger = self::getInstance();

        // Default file path if not provided
        if ($filePath === null) {
            $filePath = "./specialized-{$context}.log";
        }

        // Convert level name to Monolog's log level constant
        $level = self::LEVEL_MAP[strtoupper($levelName)] ?? self::DEBUG;

        // Create a specialized logger
        $specializedLogger = new SpecializedLogger($mainLogger, $context, $filePath, $level);

        // Cache the specialized logger
        self::$specializedLoggers[$key] = $specializedLogger;

        return $specializedLogger;
    }

    /**
     * Prevent cloning of the instance.
     */
    private function __clone()
    {
    }

    /**
     * Prevent unserialization of the instance.
     */
    public function __wakeup()
    {
    }

    /**
     * Enable or disable extended tracing to show parent caller information
     *
     * @param bool $enabled Whether to enable extended tracing
     * @return self
     */
    public function extended(bool $enabled = true): self
    {
        $this->extendedTracing = $enabled;
        return $this;
    }

    /**
     * Check if extended tracing is enabled
     *
     * @return bool
     */
    public function isExtendedTracingEnabled(): bool
    {
        return $this->extendedTracing;
    }

    /**
     * Set the depth of call chain to trace
     *
     * @param int $depth Number of callers to include in trace (minimum 1)
     * @return self
     */
    public function setExtendedTracingDepth(int $depth): self
    {
        $this->extendedTracingDepth = max(1, $depth);
        return $this;
    }

    /**
     * Determines if a log record should be processed based on filter rules
     *
     * @param array|\Monolog\LogRecord $record The log record with extra information
     * @return bool True if the record should be logged, false otherwise
     */
    private function shouldLog($record): bool
    {
        // If there are no filter rules, always log
        $hasIncludeRules = $this->hasFilterRules('include');
        $hasExcludeRules = $this->hasFilterRules('exclude');

        if (!$hasIncludeRules && !$hasExcludeRules) {
            return true;
        }

        // Handle Monolog v3 LogRecord objects
        $extra = null;
        if (is_object($record) && method_exists($record, 'extra')) {
            $extra = $record->extra;
        } elseif (is_array($record) && isset($record['extra'])) {
            $extra = $record['extra'];
        } else {
            return true; // If no extra data, we can't filter so allow it through
        }

        $file = basename($extra['file'] ?? '');
        $line = (string)($extra['line'] ?? '');
        $class = $extra['class'] ?? '';
        $method = $extra['function'] ?? '';

        // Debug log entry to help with troubleshooting filtering
        if ($class && (strpos($class, 'Router') !== false || strpos($file, 'index.php') !== false)) {
            // Log at a very low level that won't show in normal output
            error_log("[DEBUG] Filtering log record: file={$file}, class={$class}, method={$method}, rules=" . json_encode($this->filterRules));
        }

        // If include rules exist, the record must match at least one
        if ($hasIncludeRules) {
            return $this->matchesAnyRule('include', 'file', $file) ||
                $this->matchesAnyRule('include', 'line', $line) ||
                $this->matchesAnyRule('include', 'class', $class) ||
                $this->matchesAnyRule('include', 'method', $method);
        }

        // If only exclude rules exist, the record must not match any
        if ($hasExcludeRules) {
            return !($this->matchesAnyRule('exclude', 'file', $file) ||
                $this->matchesAnyRule('exclude', 'line', $line) ||
                $this->matchesAnyRule('exclude', 'class', $class) ||
                $this->matchesAnyRule('exclude', 'method', $method));
        }

        return true;
    }

    /**
     * Checks if any filter rules of the specified type exist
     *
     * @param string $type Either 'include' or 'exclude'
     * @return bool True if rules of the specified type exist
     */
    private function hasFilterRules(string $type): bool
    {
        return !empty($this->filterRules[$type]['file']) ||
            !empty($this->filterRules[$type]['line']) ||
            !empty($this->filterRules[$type]['class']) ||
            !empty($this->filterRules[$type]['method']);
    }

    /**
     * Checks if a value matches any rule in the specified filter type and category
     *
     * @param string $type The filter type ('include' or 'exclude')
     * @param string $category The category ('file', 'line', 'class', or 'method')
     * @param string $value The value to check against the rules
     * @return bool True if the value matches any rule
     */
    private function matchesAnyRule(string $type, string $category, string $value): bool
    {
        if (empty($value) || empty($this->filterRules[$type][$category])) {
            return false;
        }

        // For class names, extract the base name without namespace
        $shortValue = $value;
        if ($category === 'class' && strpos($value, '\\') !== false) {
            $shortValue = basename(str_replace('\\', '/', $value));
        }

        foreach ($this->filterRules[$type][$category] as $pattern) {
            // Check for wildcard matches
            if (strpos($pattern, '*') !== false) {
                $regex = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/i';
                if (preg_match($regex, $value) || preg_match($regex, $shortValue)) {
                    return true;
                }
            }
            // Direct string comparison with both full and short names
            elseif (strcasecmp($pattern, $value) === 0 || strcasecmp($pattern, $shortValue) === 0) {
                return true;
            }
        }

        return false;
    }

}

/**
 * A specialized logger that writes to both the main log and a dedicated file.
 */
class SpecializedLogger
{
    /**
     * Flag to enable extended call stack tracing
     *
     * @var bool
     */
    private bool $extendedTracing = false;

    /**
     * The main application logger
     *
     * @var Logger
     */
    private Logger $mainLogger;

    /**
     * The dedicated file logger for this context
     *
     * @var MonologLogger
     */
    private MonologLogger $fileLogger;

    /**
     * The context name for this specialized logger
     *
     * @var string
     */
    private string $context;

    /**
     * Constructor for the specialized logger
     *
     * @param Logger $mainLogger The main application logger
     * @param string $context The context name
     * @param string $filePath The path to the log file
     * @param int $level The log level to use
     * @throws \Exception If log file cannot be created
     */
    public function __construct(Logger $mainLogger, string $context, string $filePath, int $level)
    {
        $this->mainLogger = $mainLogger;
        $this->context = $context;

        // Create a dedicated file logger
        $this->fileLogger = new MonologLogger("specialized-{$context}");

        // Add a file handler with the custom formatter
        $handler = new StreamHandler($filePath, $level);
        $handler->setFormatter(new PrettyArrayFormatter());
        $this->fileLogger->pushHandler($handler);

        // Add context processor
        $this->fileLogger->pushProcessor(function ($record) {
            $record['extra']['specialized_context'] = $this->context;
            return $record;
        });
    }

    /**
     * Enable or disable extended tracing to show parent caller information
     *
     * @param bool $enabled Whether to enable extended tracing
     * @return self
     */
    public function extended(bool $enabled = true): self
    {
        $this->extendedTracing = $enabled;
        $this->mainLogger->extended($enabled);
        return $this;
    }



    /**
     * Check if extended tracing is enabled
     *
     * @return bool
     */
    public function isExtendedTracingEnabled(): bool
    {
        return $this->extendedTracing;
    }

    /**
     * Log a message at debug level
     *
     * @param string $message The log message
     * @param array $context The log context
     * @return void
     */
    public function debug(string $message, array $context = []): void
    {
        // Log to the specialized file
        $this->fileLogger->debug($message, $context);

        // Also log to the main logger with a prefix to identify the specialized context
        $this->mainLogger->debug("[{$this->context}] " . $message, $context);
    }

    /**
     * Log a message at info level
     *
     * @param string $message The log message
     * @param array $context The log context
     * @return void
     */
    public function info(string $message, array $context = []): void
    {
        $this->fileLogger->info($message, $context);
        $this->mainLogger->info("[{$this->context}] " . $message, $context);
    }

    /**
     * Log a message at warning level
     *
     * @param string $message The log message
     * @param array $context The log context
     * @return void
     */
    public function warning(string $message, array $context = []): void
    {
        $this->fileLogger->warning($message, $context);
        $this->mainLogger->warning("[{$this->context}] " . $message, $context);
    }

    /**
     * Log a message at error level
     *
     * @param string $message The log message
     * @param array $context The log context
     * @return void
     */
    public function error(string $message, array $context = []): void
    {
        $this->fileLogger->error($message, $context);
        $this->mainLogger->error("[{$this->context}] " . $message, $context);
    }

    /**
     * Log a message at any level
     *
     * @param string $level The log level
     * @param string $message The log message
     * @param array $context The log context
     * @return void
     */
    public function log(string $level, string $message, array $context = []): void
    {
        $this->fileLogger->log($level, $message, $context);
        $this->mainLogger->log($level, "[{$this->context}] " . $message, $context);
    }
}

class PrettyArrayFormatter implements FormatterInterface
{
    public function format(array|\Monolog\LogRecord $record): mixed
    {
        // Handle Monolog v3 LogRecord objects
        $context = [];
        $datetime = '';
        $channel = '';
        $level_name = '';
        $message = '';
        $extra = [];

        if (is_object($record)) {
            // Monolog v3 LogRecord object
            $context = $record->context;
            $datetime = $record->datetime->format('Y-m-d H:i:s');
            $channel = $record->channel;
            // Get level name from the Level object
            $level_name = $record->level->getName();
            $message = $record->message;
            $extra = $record->extra;
        } else {
            // Legacy array format
            $context = $record['context'] ?? [];
            $datetime = $record['datetime'] ?? '';
            $channel = $record['channel'] ?? '';
            $level_name = $record['level_name'] ?? '';
            $message = $record['message'] ?? '';
            $extra = $record['extra'] ?? [];
        }

        // Format arrays in the context
        if (is_array($context)) {
            foreach ($context as $key => $value) {
                if (is_array($value) && count($value) > 0) {
                    $context[$key] = "\n" . print_r($value, true);
                }
            }
        }

        // Extract filename (without path) and line number from introspection data
        $fileInfo = '';

        // Debug output to help with troubleshooting filtering
        if ((is_array($extra) && isset($extra['class']) && strpos($extra['class'] ?? '', 'Router') !== false) ||
            (is_object($extra) && isset($extra['class']) && strpos($extra['class'] ?? '', 'Router') !== false)) {
            error_log("[DEBUG] PrettyArrayFormatter - class: " . ($extra['class'] ?? 'none') . ", file: " . basename($extra['file'] ?? 'none'));
        }

        if ((is_array($extra) && isset($extra['file']) && isset($extra['line'])) ||
            (is_object($extra) && isset($extra['file']) && isset($extra['line']))) {
            // If we have call chain info, format the complete chain
            $hasCallChain = (is_array($extra) && isset($extra['call_chain']) && count($extra['call_chain']) > 1) ||
                (is_object($extra) && isset($extra['call_chain']) && count($extra['call_chain']) > 1);

            if ($hasCallChain) {
                $callChain = $extra['call_chain'] ?? [];
                $chainParts = [];

                // Build the chain from the outermost caller to the immediate caller
                // We want to show the sequence: caller -> called method -> final method
                for ($i = count($callChain) - 1; $i >= 0; $i--) {
                    $caller = $callChain[$i] ?? [];

                    $callerFile = basename($caller['file'] ?? '');
                    $callerLine = $caller['line'] ?? '';
                    $callerFunction = $caller['function'] ?? '';
                    $callerClass = $caller['class'] ?? '';

                    if (!empty($callerFile)) {
                        if (!empty($callerClass)) {
                            $callerClassname = basename(str_replace('\\', '/', $callerClass));
                            $chainParts[] = "{$callerFile} ({$callerLine}) -> {$callerClassname}->{$callerFunction}";
                        } else {
                            $chainParts[] = "{$callerFile} ({$callerLine}) -> {$callerFunction}";
                        }
                    }
                }

                // Join the chain parts to show the call sequence
                $fileInfo = " -> " . implode(" -> ", $chainParts);
            } else {
                // No extended tracing or single caller - show just the immediate caller
                $filename = basename($extra['file'] ?? '');
                $line = $extra['line'] ?? '';
                $function = $extra['function'] ?? '';
                $class = $extra['class'] ?? '';

                if (!empty($class)) {
                    $classname = basename(str_replace('\\', '/', $class));
                    $fileInfo = " -> {$filename} ({$line}) -> {$classname}->{$function}";
                } else {
                    $fileInfo = " -> {$filename} ({$line}) -> {$function}";
                }
            }

            // Add colon at the end
            $fileInfo .= ":";
        }

        /*
         * @TODO Need to add a check to make sure the value from the first array is not an array, if it is, then we need to use the second if statement to build the $context message!
         * $context is built based on the following rules :
         * $context -> Not set -> return empty string
         * $context -> Array, 1xKey with non-array Value -> return '$key => $value'
         * $context -> Array, more than 1 Key || 1xKey with array Value -> return var_export($context)
         */
        $contextStr = '';
        if (is_array($context)) {
            $firstKey = array_key_first($context) ?? null;
            if (count($context) === 1 && $firstKey !== null && !is_array($context[$firstKey])) {
                $contextStr = "Array : {$firstKey}=>{$context[$firstKey]}";
            } elseif (count($context) > 0) {
                try {
                    // Limit recursion depth to prevent infinite loops
                    $contextStr = $this->safeVarExport($context);
                } catch (\Throwable $e) {
                    $contextStr = "[Complex data structure - export failed]";
                }
            }
        }

        // Build the log message with the file info
        return $datetime . ' ' . $channel . '.' . $level_name
            . $fileInfo . ' ' . $message . ' '
            . $contextStr . "\n";
    }



    public function formatBatch(array $records): mixed
    {
        $message = '';
        foreach ($records as $record) {
            // Skip invalid records to prevent segmentation faults
            if (!is_array($record) && !($record instanceof \Monolog\LogRecord)) {
                continue;
            }
            try {
                $message .= $this->format($record);
            } catch (\Throwable $e) {
                // If formatting fails, skip this record
                continue;
            }
        }

        return $message;
    }

    /**
     * Safely export a variable to a string representation with limited recursion
     * 
     * @param mixed $var The variable to export
     * @param int $maxDepth Maximum recursion depth
     * @param int $currentDepth Current recursion depth
     * @param array $seenObjects Objects that have already been processed (to prevent circular references)
     * @return string The exported variable as a string
     */
    private function safeVarExport($var, int $maxDepth = 3, int $currentDepth = 0, array &$seenObjects = []): string
    {
        // Check recursion depth
        if ($currentDepth > $maxDepth) {
            return '[Max recursion depth reached]';
        }

        // Handle different variable types
        switch (gettype($var)) {
            case 'NULL':
                return 'NULL';
            case 'boolean':
                return $var ? 'true' : 'false';
            case 'integer':
            case 'double':
                return (string)$var;
            case 'string':
                return "'" . addslashes($var) . "'";
            case 'array':
                // Start building array string
                $result = "array (\n";
                $indent = str_repeat('  ', $currentDepth + 1);

                foreach ($var as $key => $value) {
                    $result .= $indent . $this->safeVarExport($key, $maxDepth, $currentDepth + 1, $seenObjects) . ' => ';
                    $result .= $this->safeVarExport($value, $maxDepth, $currentDepth + 1, $seenObjects) . ",\n";
                }

                return $result . str_repeat('  ', $currentDepth) . ')';
            case 'object':
                // Handle circular references
                $hash = spl_object_hash($var);
                if (isset($seenObjects[$hash])) {
                    return '[Circular reference to ' . get_class($var) . ']';
                }
                $seenObjects[$hash] = true;

                // Get object properties
                $reflection = new \ReflectionObject($var);
                $properties = $reflection->getProperties();

                $result = get_class($var) . "::__set_state(array(\n";
                $indent = str_repeat('  ', $currentDepth + 1);

                foreach ($properties as $property) {
                    $property->setAccessible(true);
                    $propName = $property->getName();
                    $propValue = $property->getValue($var);

                    $result .= $indent . "'" . $propName . "' => ";
                    $result .= $this->safeVarExport($propValue, $maxDepth, $currentDepth + 1, $seenObjects) . ",\n";
                }

                return $result . str_repeat('  ', $currentDepth) . '))';
            case 'resource':
                return '[Resource]';
            default:
                return '[Unknown type]';
        }
    }
}
