<?php

namespace AmgGroup\Tests;

use AmgGroup\PrettyArrayFormatter;
use Monolog\Level;
use Monolog\LogRecord;
use PHPUnit\Framework\TestCase;

/**
 * Test case for the PrettyArrayFormatter class
 */
class PrettyArrayFormatterTest extends TestCase
{
    /**
     * Test formatting an array record (Monolog v1/v2 style)
     */
    public function testFormatArrayRecord(): void
    {
        $formatter = new PrettyArrayFormatter();
        
        $record = [
            'message' => 'Test message',
            'context' => ['key' => 'value'],
            'level' => 200, // INFO level
            'level_name' => 'INFO',
            'channel' => 'test',
            'datetime' => '2023-01-01 12:00:00',
            'extra' => [
                'file' => '/path/to/file.php',
                'line' => 123,
                'class' => 'TestClass',
                'function' => 'testMethod'
            ]
        ];
        
        $result = $formatter->format($record);
        
        // Check that the result contains the expected elements
        $this->assertStringContainsString('2023-01-01 12:00:00', $result);
        $this->assertStringContainsString('test.INFO', $result);
        $this->assertStringContainsString('file.php (123)', $result);
        $this->assertStringContainsString('TestClass->testMethod', $result);
        $this->assertStringContainsString('Test message', $result);
        $this->assertStringContainsString('key=>value', $result);
    }
    
    /**
     * Test formatting a LogRecord object (Monolog v3 style)
     */
    public function testFormatLogRecordObject(): void
    {
        // Skip this test if Monolog v3 classes are not available
        if (!class_exists('Monolog\LogRecord')) {
            $this->markTestSkipped('Monolog v3 LogRecord class not available');
            return;
        }
        
        $formatter = new PrettyArrayFormatter();
        
        // Create a DateTime object for the record
        $datetime = new \DateTimeImmutable('2023-01-01 12:00:00');
        
        // Create a LogRecord object
        $record = new LogRecord(
            $datetime,
            'test',
            Level::Info,
            'Test message',
            ['key' => 'value'],
            [
                'file' => '/path/to/file.php',
                'line' => 123,
                'class' => 'TestClass',
                'function' => 'testMethod'
            ]
        );
        
        $result = $formatter->format($record);
        
        // Check that the result contains the expected elements
        $this->assertStringContainsString('2023-01-01 12:00:00', $result);
        $this->assertStringContainsString('test.INFO', $result);
        $this->assertStringContainsString('file.php (123)', $result);
        $this->assertStringContainsString('TestClass->testMethod', $result);
        $this->assertStringContainsString('Test message', $result);
        $this->assertStringContainsString('key=>value', $result);
    }
    
    /**
     * Test formatting a batch of records
     */
    public function testFormatBatch(): void
    {
        $formatter = new PrettyArrayFormatter();
        
        $records = [
            [
                'message' => 'Test message 1',
                'context' => ['key1' => 'value1'],
                'level' => 200, // INFO level
                'level_name' => 'INFO',
                'channel' => 'test',
                'datetime' => '2023-01-01 12:00:00',
                'extra' => [
                    'file' => '/path/to/file1.php',
                    'line' => 123,
                    'class' => 'TestClass1',
                    'function' => 'testMethod1'
                ]
            ],
            [
                'message' => 'Test message 2',
                'context' => ['key2' => 'value2'],
                'level' => 300, // WARNING level
                'level_name' => 'WARNING',
                'channel' => 'test',
                'datetime' => '2023-01-01 12:01:00',
                'extra' => [
                    'file' => '/path/to/file2.php',
                    'line' => 456,
                    'class' => 'TestClass2',
                    'function' => 'testMethod2'
                ]
            ]
        ];
        
        $result = $formatter->formatBatch($records);
        
        // Check that the result contains both messages
        $this->assertStringContainsString('Test message 1', $result);
        $this->assertStringContainsString('Test message 2', $result);
        $this->assertStringContainsString('file1.php (123)', $result);
        $this->assertStringContainsString('file2.php (456)', $result);
    }
    
    /**
     * Test formatting with extended tracing (call chain)
     */
    public function testFormatWithCallChain(): void
    {
        $formatter = new PrettyArrayFormatter();
        
        $record = [
            'message' => 'Test message',
            'context' => [],
            'level' => 200,
            'level_name' => 'INFO',
            'channel' => 'test',
            'datetime' => '2023-01-01 12:00:00',
            'extra' => [
                'file' => '/path/to/file.php',
                'line' => 123,
                'class' => 'TestClass',
                'function' => 'testMethod',
                'call_chain' => [
                    [
                        'file' => '/path/to/file1.php',
                        'line' => 100,
                        'class' => 'CallerClass1',
                        'function' => 'callerMethod1'
                    ],
                    [
                        'file' => '/path/to/file2.php',
                        'line' => 200,
                        'class' => 'CallerClass2',
                        'function' => 'callerMethod2'
                    ]
                ]
            ]
        ];
        
        $result = $formatter->format($record);
        
        // Check that the call chain is included in the output
        $this->assertStringContainsString('file2.php (200)', $result);
        $this->assertStringContainsString('CallerClass2->callerMethod2', $result);
        $this->assertStringContainsString('file1.php (100)', $result);
        $this->assertStringContainsString('CallerClass1->callerMethod1', $result);
    }
}