# AMG Logger System Documentation

## Overview

The AMG Logger system is a sophisticated logging implementation built on top of the Monolog library. It's designed as a singleton to ensure consistent logging throughout the application, with support for specialized loggers, extended call tracing, and custom formatting.

## Core Components

### 1. Logger Class

The main `Logger` class extends Monolog's `MonologLogger` and implements a singleton pattern for application-wide logging.

#### Key Properties

- `$instance`: Static property holding the singleton instance
- `$specializedLoggers`: Cache of specialized loggers for different contexts
- `$extendedTracing`: Flag to enable/disable detailed call stack tracing
- `$extendedTracingDepth`: Controls how many levels of the call stack to include
- `$filterRules`: Contains include and exclude rules for filtering log entries

#### Initialization

The private constructor:
1. Loads configuration from the Config singleton
2. Sets up handlers based on configuration (file, console, error log)
3. Configures error handling if enabled
4. Sets up processors for introspection, stack traces, process ID, and memory usage

#### Call Tracing Implementation

The introspection processor:
1. Gets the debug backtrace with `debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS)`
2. Skips internal logger classes by checking class names (Monolog, Amg\Logger, etc.)
3. Finds the first non-logger class in the stack - the immediate caller
4. Stores basic caller information (file, line, class, function)
5. When extended tracing is enabled, it captures additional callers up to the configured depth

### 2. SpecializedLogger Class

A wrapper that writes log entries to both the main log and a context-specific log file.

#### Key Features

- Maintains its own file handler for context-specific logs
- Prefixes log messages with the context when writing to the main log
- Delegates to the main logger's extended tracing functionality

### 3. PrettyArrayFormatter Class

A custom formatter that creates human-readable log entries with detailed caller information.

#### Format Method Breakdown

1. **Context Formatting**:
    - Formats array values in the context for better readability
    - Uses `print_r()` for multi-element arrays

2. **Call Stack Information Extraction**:
    - Checks if basic file/line information is available
    - If extended tracing is enabled, processes the call chain:
        - Iterates through the call chain from outermost caller to immediate caller
        - Extracts file, line, class, and function for each caller
        - Formats each caller as `filename (line) -> Class->method`
        - Joins the chain with ` -> ` separators
    - If extended tracing is disabled, shows only the immediate caller

3. **Context Value Formatting**:
    - For single-element, non-array contexts: formats as `Array: key=>value`
    - For multi-element or array-value contexts: uses `var_export()`
    - For empty contexts: returns an empty string

4. **Final Log Entry Assembly**:
    - Combines timestamp, channel, level, call information, message, and context
    - Format: `datetime channel.LEVEL -> file (line) -> Class->method: message context`

## Usage Examples

### Basic Logging

```php
$logger = Logger::getInstance();
$logger->info("User logged in", ["userId" => 123]);
```

### Extended Call Tracing

```php
$logger->extended(true);
$logger->info("Processing transaction", ["transactionId" => "abc123"]);
$logger->extended(false);
```

### Specialized Logging

```php
$auditLogger = Logger::getSpecializedLogger("audit", "./logs/audit.log");
$auditLogger->info("Policy updated", ["policyId" => "xyz789"]);
```

## Call Chain Visualization

When extended tracing is enabled, the log format shows the entire call chain:

```
timestamp app.INFO -> Main.php (806) -> MSGraph->updateConditionalAccessPolicy (263): Message content
```

This indicates:
1. Log was generated at `line 263` in the `updateConditionalAccessPolicy` method of the `MSGraph` class
2. The method was called from `line 806` in `Main.php`

The chain can be more complex with deeper call stacks:

```
timestamp app.INFO -> Controller.php (52) -> Service.php (124) -> Repository.php (87): Message content
```

## Trace Extraction Process

1. **Initial Stack Collection**: `debug_backtrace()` returns the full call stack
2. **Logger Frame Skipping**: Frames from logger classes are skipped
3. **Immediate Caller**: First non-logger frame is identified as the immediate caller
4. **Extended Chain Building**: With extended tracing, subsequent frames are processed
5. **Reverse Order Presentation**: Chain is presented from outermost caller to immediate caller

This implementation allows developers to trace the execution path that led to a log entry, making debugging and understanding application flow much easier.

## Log Filtering System

The Logger implements a filtering system to include or exclude log entries based on their source.

### Configuration

Filter rules are defined in the configuration file under the `debug.include` and `debug.exclude` sections:

```yaml
debug:
  exclude:
    file: ["unwanted_file.php", "*test*.php"]
    class: ["SomeClass", "Debug*"]
    method: ["verbose*"]
    line: ["42"]
  include:
    file: ["important.php"]
    class: ["CriticalClass"]
    method: ["processPayment"]
    line: []
```

### Filtering Logic

1. **No Filter Rules**: If no filter rules are defined, all log entries are processed
2. **Include Rules Only**: If only include rules exist, only log entries matching at least one rule are processed
3. **Exclude Rules Only**: If only exclude rules exist, log entries matching any rule are discarded
4. **Both Include and Exclude**: Include rules take precedence; only entries matching include rules are processed

### Wildcard Support

Filter patterns support wildcards using the `*` character:
- `*Controller.php` - matches any file ending with "Controller.php"
- `Debug*` - matches any class starting with "Debug"
- `get*User` - matches any method starting with "get" and ending with "User"

### Internal Process

Filtering happens in the introspection processor:
1. Caller information is extracted from the backtrace
2. The `shouldLog()` method evaluates whether the log entry matches filter rules
3. Entries that should be filtered out are given a log level lower than any defined handler

## Testing

The AMG Logger system includes a comprehensive test suite to ensure all components work as expected.

### Running Tests

To run the test suite, use the following command:

```bash
composer test
```

This will execute all tests in the `tests` directory using PHPUnit.

### Test Structure

The test suite includes tests for:

1. **Logger Class** - Tests the singleton pattern, extended tracing, and specialized logger creation
2. **SpecializedLogger Class** - Tests logging to both main and specialized log files
3. **PrettyArrayFormatter** - Tests formatting of log records with various configurations

### Writing New Tests

When adding new features to the logger, please add corresponding tests in the `tests` directory following the existing pattern:

1. Create a test class that extends `PHPUnit\Framework\TestCase`
2. Implement test methods that verify the behavior of your feature
3. Run the tests to ensure everything works as expected
