<?php

namespace AmgGroup;
/**
 * LdapAuthenticator - A class to handle LDAP authentication against Active Directory
 */
class LdapAuthenticator
{
    /**
     * Holds the single instance of Database.
     *
     * @var LdapAuthenticator|null
     */
    private static ?LdapAuthenticator $instance = null;

    /**
     * @var string The domain FQDN (e.g., 'example.local')
     */
    private string $domainFqdn;

    /**
     * @var string The LDAP server address
     */
    private string $ldapServer;

    /**
     * @var resource|false The LDAP connection
     */
    private $connection = false;

    /**
     * @var string|null Last error message
     */
    private ?string $lastError = null;

    /**
     * @var array Base DNs to search for users
     */
    private array $baseDns = [];
    private \AmgGroup\Config $config;
    private \AmgGroup\Logger $logger;

    /**
     * Constructor
     *
     * @param string $domainFqdn The domain FQDN
     * @param string $ldapServer The LDAP server address
     */
    private function __construct()
    {
        // Configure any global site data
        $this->config = \AmgGroup\Config::getInstance();
        $this->logger = \AmgGroup\Logger::getInstance();

        $this->domainFqdn = $this->config->get('authentication.DOMAIN_FQDN') ?? '';
        $this->ldapServer = $this->config->get('authentication.LDAP_SERVER') ?? '';

        // Build the base DNs from the domain FQDN
        $domainParts = explode('.', $this->domainFqdn);
        $dcString = 'DC=' . implode(',DC=', $domainParts);

        $this->baseDns = [
            $dcString
        ];
    }

    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }


    /**
     * Authenticate a user against LDAP
     *
     * @param string $username The username (can be username, domain\username, or user@domain.com)
     * @param string $password The password
     * @return bool True if authentication successful, false otherwise
     */
    public function authenticate(string $username, string $password): bool
    {
        // Clear any previous errors
        $this->lastError = null;

        // Check if the username is already in UPN format (contains @)
        if (strpos($username, '@') !== false) {
            $userDn = $username; // Already in the format user@domain.com
        } else {
            // Clean input
            $username = $this->sanitizeUsername($username);

            // Check if the username includes a domain prefix (DOMAIN\user)
            if (strpos($username, '\\') !== false) {
                // Split the domain and username
                list($domain, $username) = explode('\\', $username, 2);
                $username = $this->sanitizeUsername($username);
            }

            // Format username with domain (UPN format)
            $userDn = $username . '@' . $this->domainFqdn;
        }

        // Connect to LDAP server
        if (!$this->connect()) {
            return false;
        }

        // Try to bind with the user credentials
        $bind = @ldap_bind($this->connection, $userDn, $password);

        // Check for password expiration or other extended errors
        ldap_get_option($this->connection, LDAP_OPT_DIAGNOSTIC_MESSAGE, $extendedError);
        if (!empty($extendedError)) {
            $errno = $this->parseExtendedError($extendedError);
            if ($errno == 532) {
                $this->lastError = 'Unable to login: Password expired';
                return false;
            }
        }

        // If binding failed
        if (!$bind) {
            $this->lastError = 'Authentication failed: ' . ldap_error($this->connection);
            return false;
        }

        // Verify the user exists in one of the search bases
        if (!$this->verifyUserExists()) {
            return false;
        }

        // Close the LDAP connection
        $this->disconnect();

        return true;
    }

    /**
     * Get the last error message
     *
     * @return string|null The last error message or null if no error
     */
    public function getLastError(): ?string
    {
        return $this->lastError;
    }

    /**
     * Connect to the LDAP server
     *
     * @return bool True if connection successful, false otherwise
     */
    private function connect(): bool
    {
        $this->connection = ldap_connect("ldap://" . $this->ldapServer . "/");

        if (!$this->connection) {
            $this->lastError = 'Could not connect to LDAP server';
            return false;
        }

        // Set LDAP options
        ldap_set_option($this->connection, LDAP_OPT_PROTOCOL_VERSION, 3);
        ldap_set_option($this->connection, LDAP_OPT_REFERRALS, 0);

        return true;
    }

    /**
     * Disconnect from the LDAP server
     */
    private function disconnect(): void
    {
        if ($this->connection) {
            ldap_close($this->connection);
            $this->connection = false;
        }
    }

    /**
     * Parse the extended error from LDAP
     *
     * @param string $extendedError The extended error message
     * @return int The error code
     */
    private function parseExtendedError(string $extendedError): int
    {
        $errno = explode(',', $extendedError);
        $errno = $errno[2] ?? '';
        $errno = explode(' ', $errno);
        $errno = $errno[2] ?? 0;

        return intval($errno);
    }

    /**
     * Verify the user exists in one of the base DNs
     *
     * @return bool True if user exists, false otherwise
     */
    private function verifyUserExists(): bool
    {
        $connections = array_fill(0, count($this->baseDns), $this->connection);
        $result = ldap_search($connections, $this->baseDns, "(cn=*)");

        if (!count($result)) {
            $this->lastError = 'User search failed: ' . ldap_error($this->connection);
            return false;
        }

        return true;
    }

    /**
     * Sanitize the username
     *
     * @param string $username The username to sanitize
     * @return string The sanitized username
     */
    private function sanitizeUsername(string $username): string
    {
        // Remove any potentially harmful tags
        $username = strip_tags(trim($username));

        // If the username contains a domain part (user@domain.com), preserve it
        if (strpos($username, '@') !== false) {
            return $username;
        }

        return $username;
    }

    /**
     * Check if a user is a member of a specified group
     *
     * @param string $username The username to check
     * @param string $groupName The group name to check membership for
     * @return bool True if the user is a member of the group, false otherwise
     */
    public function isInGroup(string $username, string $groupName): bool
    {

        $this->lastError = null;

        // Clean input, but preserve @ and domain part if present
        if (strpos($username, '@') === false && strpos($username, '\\') === false) {
            $username = $this->sanitizeUsername($username);
        }
        $groupName = trim($groupName);
        
        // Connect to LDAP server
        if (!$this->connect()) {
            $this->logger->error("Could not connect to LDAP server");
            return false;
        }
        
        // For this operation, we need to bind as an administrator or service account
        // that has permissions to query group memberships
        if (!$this->bindAsServiceAccount()) {
            $this->logger->error("Could not bind as service account");
            return false;
        }
        
        try {
            // First, find the user's DN
            $userDn = $this->findUserDn($username);
            if (!$userDn) {
                $this->logger->error("UserDn not found");
                $this->lastError = "User not found";
                return false;
            }
            //$this->logger->info("UserDn found", [$userDn]);
            // Next, find the group's DN
            //$groupDn = $this->findGroupDn($groupName);
            $groupDn = $this->findGroupDnDomainWide($groupName);
            if (!$groupDn) {
                $this->logger->error("GroupDn not found for group $groupName");;
                $this->lastError = "Group not found";
                return false;
            }
            //$this->logger->info("GroupDn found", [$groupDn]);
            // Check if the user is a member of the group
            $result = $this->checkGroupMembership($userDn, $groupDn);
            //$this->logger->info("Group membership check result", [$result]);
            $this->disconnect();
            return $result;
        } catch (\Exception $e) {
            $this->lastError = "Error checking group membership: " . $e->getMessage();
            $this->disconnect();
            return false;
        }
    }
    
    /**
     * Bind to LDAP using service account credentials
     * This account needs sufficient privileges to search for users and groups
     * 
     * @return bool True if binding was successful, false otherwise
     */
    private function bindAsServiceAccount(): bool
    {
        // You could store these credentials in your config and retrieve them
        $serviceUsername = $this->config->get('authentication.SERVICE_USERNAME') ?? '';
        $servicePassword = $this->config->get('authentication.SERVICE_PASSWORD') ?? '';
        
        if (empty($serviceUsername) || empty($servicePassword)) {
            $this->lastError = 'Service account credentials not configured';
            return false;
        }
        
        // Format service account username with domain
        if (strpos($serviceUsername, '@') === false) {
            $serviceDn = $serviceUsername . '@' . $this->domainFqdn;
        }
        $serviceDn = $serviceUsername;
        //$this->logger->info("Service account DN: $serviceDn");
        // Try to bind with service account credentials
        $bind = @ldap_bind($this->connection, $serviceDn, $servicePassword);
        
        if (!$bind) {
            $this->lastError = 'Service account authentication failed: ' . ldap_error($this->connection);
            return false;
        }
        
        return true;
    }

    /**
     * Find a user's DN in Active Directory
     *
     * @param string $username The username to find (can be username, domain\username, or user@domain.com)
     * @return string|null The user's DN if found, null otherwise
     */
    private function findUserDn(string $username): ?string
    {
        //$this->logger->info("Finding user DN for username $username");
        // Check if the username is in UPN format (user@domain.com)
        if (strpos($username, '@') !== false) {
            $filter = "(userPrincipalName=$username)";
        } else {
            // Check if the username includes a domain prefix (DOMAIN\user)
            if (strpos($username, '\\') !== false) {
                // Split the domain and username
                list($domain, $username) = explode('\\', $username, 2);
                $username = $this->sanitizeUsername($username);
            }

            // Create the search filter for sAMAccountName
            $filter = "(sAMAccountName=$username)";
        }

        // Search in all configured base DNs
        foreach ($this->baseDns as $baseDn) {
            $search = ldap_search($this->connection, $baseDn, $filter);

            if ($search && ldap_count_entries($this->connection, $search) > 0) {
                $entry = ldap_first_entry($this->connection, $search);
                return ldap_get_dn($this->connection, $entry);
            }
        }

        return null;
    }
    
    /**
     * Find a group's DN in Active Directory
     *
     * @param string $groupName The group name to find
     * @return string|null The group's DN if found, null otherwise
     */
    private function findGroupDn(string $groupName): ?string
    {
        // Get domain components for the base DN
        $domainParts = explode('.', $this->domainFqdn);
        $dcString = 'DC=' . implode(',DC=', $domainParts);
        
        // Common locations for groups in Active Directory
        $groupLocations = [
            "CN=Groups,$dcString",
            "OU=Groups,$dcString",
            "OU=Security Groups,$dcString"
        ];
        
        // Create the search filter - we'll look for the cn (common name) attribute
        $filter = "(cn=$groupName)";
        
        // Search in all potential group locations
        foreach ($groupLocations as $location) {
            $search = @ldap_search($this->connection, $location, $filter);
            
            if ($search && ldap_count_entries($this->connection, $search) > 0) {
                $entry = ldap_first_entry($this->connection, $search);
                return ldap_get_dn($this->connection, $entry);
            }
        }
        
        return null;
    }
    
    /**
     * Check if a user is a member of a group
     *
     * @param string $userDn The user's DN
     * @param string $groupDn The group's DN
     * @return bool True if the user is a member of the group, false otherwise
     */
    private function checkGroupMembership(string $userDn, string $groupDn): bool
    {
        //$this->logger->info("Checking group membership for user $userDn and group $groupDn");
        // Method 1: Direct membership check using the member attribute
        $filter = "(&(objectClass=group)(distinguishedName=$groupDn)(member=$userDn))";
        $search = ldap_search($this->connection, $groupDn, $filter);
        
        if ($search && ldap_count_entries($this->connection, $search) > 0) {
            return true;
        }
        
        // Method 2: Nested group membership check using the memberOf attribute on the user
        $filter = "(&(objectClass=user)(distinguishedName=$userDn)(memberOf:1.2.840.113556.1.4.1941:=$groupDn))";
        $domainParts = explode('.', $this->domainFqdn);
        $baseDn = 'DC=' . implode(',DC=', $domainParts);
        $search = ldap_search($this->connection, $baseDn, $filter);
        
        return $search && ldap_count_entries($this->connection, $search) > 0;
    }

    private function findGroupDnDomainWide(string $groupName): ?string
    {
        // Get domain components for the base DN
        $domainParts = explode('.', $this->domainFqdn);
        $baseDn = 'DC=' . implode(',DC=', $domainParts);

        // Create the search filter - search by sAMAccountName or CN
        $filter = "(&(objectClass=group)(|(cn=$groupName)(sAMAccountName=$groupName)))";

        // Perform domain-wide search
        //$this->logger->debug("Performing domain-wide search for group: $groupName");
        $search = @ldap_search($this->connection, $baseDn, $filter);

        if ($search && ldap_count_entries($this->connection, $search) > 0) {
            $entry = ldap_first_entry($this->connection, $search);
            $dn = ldap_get_dn($this->connection, $entry);
            //$this->logger->info("Found group: $dn");
            return $dn;
        }

        $this->logger->error("Group not found: $groupName");
        return null;
    }


    /**
     * Set custom base DNs (optional)
     *
     * @param array $baseDns The base DNs to search for users
     * @return self
     */
    public function setBaseDns(array $baseDns): self
    {
        $this->baseDns = $baseDns;
        return $this;
    }

    /**
     * Add a base DN to the search list
     *
     * @param string $baseDn The base DN to add
     * @return self
     */
    public function addBaseDn(string $baseDn): self
    {
        $this->baseDns[] = $baseDn;
        return $this;
    }

    /**
     * Destructor - ensure connection is closed
     */
    public function __destruct()
    {
        $this->disconnect();
    }
}