<?php

namespace AmgGroup\Drivers;

use AmgGroup\ConfigDriverInterface;

class IniDriver implements ConfigDriverInterface
{
    public function supports(string $path): bool
    {
        return str_ends_with(strtolower($path), '.ini');
    }

    public function load(string $path): array
    {
        $data = parse_ini_file($path, true, INI_SCANNER_TYPED);
        return is_array($data) ? $data : [];
    }

    public function dump(array $data, string $path): void
    {
        $globals = '';
        $sections = '';

        // Write global keys first so they are not captured by a preceding section
        foreach ($data as $key => $value) {
            if (!is_array($value)) {
                $globals .= $key . '=' . self::iniScalar($value) . "\n";
            }
        }

        // Then write sections
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $sections .= "[{$key}]\n";
                foreach ($value as $k => $v) {
                    $sections .= $k . '=' . self::iniScalar($v) . "\n";
                }
            }
        }

        $out = $globals . ($sections !== '' ? ($globals !== '' ? "\n" : '') . $sections : '');

        if (file_put_contents($path, $out) === false) {
            throw new \RuntimeException("Failed to write INI config: {$path}");
        }
    }

    private static function iniScalar(mixed $v): string
    {
        if (is_bool($v)) return $v ? 'true' : 'false';
        if (is_numeric($v)) return (string)$v;
        return (string)$v;
    }
}
