### AMG Config System

A lightweight, DI‑friendly configuration system with pluggable file format drivers. It loads one or more config files (YAML, JSON, INI), merges them in order, exposes the result via a simple API and `ArrayAccess`, and can save the merged configuration back to a chosen file/format.

#### Components
- `AmgGroup\ConfigInterface`
  - `get(string $key, mixed $default = null): mixed`
  - `set(string $key, mixed $value): void`

- `AmgGroup\Config` (implements `ConfigInterface`, `ArrayAccess`)
  - `loadFromFiles(string ...$paths): void`
  - `saveTo(string $path): void`
  - `get(...)`, `set(...)`
  - Array-style access: `$config['key']`
  - Optional static singletons for legacy/transition: `Config::getInstance()`, `Config::setInstance(ConfigInterface $instance)`

- `AmgGroup\ConfigDriverInterface`
  - `supports(string $path): bool` — whether the driver handles the given file path (usually by extension)
  - `load(string $path): array` — read and parse into an array
  - `dump(array $data, string $path): void` — serialize and write to disk

- `AmgGroup\DriverRegistry`
  - Holds an array of drivers and resolves the correct one for a target path.

- Built-in drivers (`AmgGroup\Drivers`)
  - `YamlDriver` (uses `symfony/yaml`)
  - `JsonDriver` (native JSON)
  - `IniDriver` (native INI parsing, simple dumper)

---

### Quick start

```php
use AmgGroup\Config;
use AmgGroup\DriverRegistry;
use AmgGroup\Drivers\{YamlDriver, JsonDriver, IniDriver};

$registry = new DriverRegistry([
    new YamlDriver(),
    new JsonDriver(),
    new IniDriver(),
]);

$config = new Config($registry);

// Load defaults, then env, then local (later files override earlier ones)
$config->loadFromFiles('/path/config.default.yaml', '/path/config.env.json', '/path/config.local.ini');

// Read values
$db = $config->get('db', []);
$debug = $config['debug'] ?? false; // ArrayAccess

// Modify values
$config->set('featureX', true);

// Save merged configuration as JSON
$config->saveTo('/path/out/config.json');
```

---

### Loading multiple files and merge order

- Call `loadFromFiles($file1, $file2, ...)` with files in priority order from lowest to highest.
- Internally the data is merged using `array_replace_recursive`. Keys from later files override earlier ones.
- Missing files are skipped with a warning (if a logger is in use) and do not throw exceptions.

Example:

```php
// defaults.yaml -> env.json -> local.ini
$config->loadFromFiles($defaults, $env, $local);
// Later keys win
```

---

### Accessing and modifying configuration

- Get a value: `get('key', $default)` or `$config['key']`.
- Set a value: `set('key', $value)` or `$config['key'] = $value`.
- Values can be arrays (e.g., `db` section), scalars, booleans, etc.

```php
$host = $config->get('db')['host'] ?? 'localhost';
$config->set('debug', true);
$enabled = $config['featureX'] ?? false; // ArrayAccess
```

---

### Saving configuration

Save the current merged configuration to any supported format by providing a target path with a known extension.

```php
$config->saveTo('/tmp/app-config.yaml');
$config->saveTo('/tmp/app-config.json');
$config->saveTo('/tmp/app-config.ini');
```

The `DriverRegistry` selects the right driver based on the path. If no driver supports the path, an `InvalidArgumentException` is thrown.

Note: Saving writes a full merged snapshot to the chosen file. It does not try to split changes back into the original source files.

---

### Using the optional global instance

For legacy code or simple scripts, you may use the static singleton accessor:

```php
// Bootstrap once
Config::setInstance($config);

// Later, elsewhere
$cfg = Config::getInstance();
$debug = $cfg->get('debug', false);
```

Prefer constructor/DI injection in new code; use the singleton only as a transition.

---

### Supported formats and examples

#### YAML

Example `config.yaml`:

```yaml
db:
  host: localhost
  port: 5432
debug: true
```

Load and use:

```php
$registry = new DriverRegistry([new AmgGroup\Drivers\YamlDriver()]);
$config = new Config($registry);
$config->loadFromFiles('config.yaml');
$debug = $config->get('debug');
```

#### JSON

Example `config.json`:

```json
{
  "service": {"url": "https://example.com", "timeout": 30},
  "enabled": false
}
```

Load and use:

```php
$registry = new DriverRegistry([new AmgGroup\Drivers\JsonDriver()]);
$config = new Config($registry);
$config->loadFromFiles('config.json');
$url = $config->get('service')["url"] ?? null;
```

#### INI

Example `config.ini`:

```ini
[section]
a=1
b=true
flag=false
```

Load and use:

```php
$registry = new DriverRegistry([new AmgGroup\Drivers\IniDriver()]);
$config = new Config($registry);
$config->loadFromFiles('config.ini');
$flag = $config->get('flag', false);
```

---

### Extending with a new driver

Implement `ConfigDriverInterface` and register it:

```php
use AmgGroup\ConfigDriverInterface;

final class XmlDriver implements ConfigDriverInterface
{
    public function supports(string $path): bool
    { return str_ends_with(strtolower($path), '.xml'); }

    public function load(string $path): array
    {
        $xml = simplexml_load_file($path, \SimpleXMLElement::class);
        return $xml ? json_decode(json_encode($xml), true) ?? [] : [];
    }

    public function dump(array $data, string $path): void
    { file_put_contents($path, '<config />'); }
}

$registry = new DriverRegistry([
    new YamlDriver(), new JsonDriver(), new IniDriver(), new XmlDriver(),
]);
```

---

### Error handling

- Missing files: skipped during `loadFromFiles` (a warning is logged via `LoggerInterface` if one is provided to `Config`).
- YAML: parse errors from `symfony/yaml` bubble as exceptions.
- JSON: read/encode/decode failures throw `RuntimeException` with details.
- INI: `parse_ini_file` returns arrays; write errors throw `RuntimeException`.
- Saving: if no driver supports the target path, `DriverRegistry::resolve` throws `InvalidArgumentException`.

---

### Testing

Run the included tests:

```bash
./vendor/bin/phpunit --no-configuration --bootstrap vendor/autoload.php tests --testdox
```

Covers:

- Driver round-trip (dump then load) for YAML/JSON/INI
- Config merge order and overrides
- Saving merged config to JSON

---

### Notes

- The YAML driver uses `symfony/yaml` which is available in `vendor/symfony/yaml` in this project.
- The system writes a single, canonical merged output when saving; it does not attempt per-source round-tripping.
