<?php

namespace AmgGroup\Tests;

use AmgGroup\Config;
use AmgGroup\DriverRegistry;
use AmgGroup\Drivers\IniDriver;
use AmgGroup\Drivers\JsonDriver;
use AmgGroup\Drivers\YamlDriver;
use PHPUnit\Framework\TestCase;

class ConfigTest extends TestCase
{
    private function makeConfig(): Config
    {
        $registry = new DriverRegistry([
            new YamlDriver(),
            new JsonDriver(),
            new IniDriver(),
        ]);
        return new Config($registry);
    }

    public function testMergeOrderAndGetSet(): void
    {
        $cfg = $this->makeConfig();

        // Prepare three files: default (yaml), env (json), local (ini)
        $yamlDriver = new YamlDriver();
        $jsonDriver = new JsonDriver();
        $iniDriver = new IniDriver();

        $defaults = [
            'db' => ['host' => 'localhost', 'port' => 3306],
            'debug' => false,
            'featureX' => false,
        ];
        $env = [
            'db' => ['host' => 'db.example', 'port' => 3306],
            'featureX' => true,
        ];
        $local = [
            'db' => ['port' => 3307],
        ];

        $f1 = tempnam(sys_get_temp_dir(), 'cfg') . '.yaml';
        $f2 = tempnam(sys_get_temp_dir(), 'cfg') . '.json';
        $f3 = tempnam(sys_get_temp_dir(), 'cfg') . '.ini';

        $yamlDriver->dump($defaults, $f1);
        $jsonDriver->dump($env, $f2);
        $iniDriver->dump($local, $f3);

        $cfg->loadFromFiles($f1, $f2, $f3);

        // Later files override earlier ones
        $this->assertSame('db.example', $cfg->get('db')['host']);
        $this->assertSame(3307, $cfg->get('db')['port']);
        $this->assertSame(true, $cfg->get('featureX'));
        $this->assertSame(false, $cfg['debug']); // ArrayAccess

        // set should override value
        $cfg->set('debug', true);
        $this->assertTrue($cfg->get('debug'));

        @unlink($f1);
        @unlink($f2);
        @unlink($f3);
    }

    public function testSaveToJson(): void
    {
        $cfg = $this->makeConfig();
        $cfg->set('a', 1);
        $cfg->set('b', ['x' => true]);

        $out = tempnam(sys_get_temp_dir(), 'cfg') . '.json';
        $cfg->saveTo($out);

        $json = new JsonDriver();
        $read = $json->load($out);
        $this->assertSame(['a' => 1, 'b' => ['x' => true]], $read);

        @unlink($out);
    }
}
