<?php

namespace AmgGroup;

use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

class Config implements ConfigInterface, \ArrayAccess
{
    private array $data = [];
    private LoggerInterface $logger;
    private DriverRegistry $drivers;

    private static ?ConfigInterface $instance = null;

    public function __construct(DriverRegistry $drivers, ?LoggerInterface $logger = null)
    {
        $this->drivers = $drivers;
        $this->logger = $logger ?? new NullLogger();
    }

    public static function getInstance(): ConfigInterface
    {
        if (!self::$instance) {
            // Provide a minimal default with no drivers; most users should setInstance during bootstrap
            self::$instance = new self(new DriverRegistry([]));
        }
        return self::$instance;
    }

    public static function setInstance(ConfigInterface $instance): void
    {
        self::$instance = $instance;
    }

    // Loading and saving
    public function loadFromFiles(string ...$paths): void
    {
        foreach ($paths as $path) {
            if (!$path) { continue; }
            if (!is_file($path)) {
                $this->logger->warning("Config file missing: {$path}");
                continue;
            }
            $driver = $this->drivers->resolve($path);
            $arr = $driver->load($path);
            if (!is_array($arr)) { $arr = []; }
            $this->data = array_replace_recursive($this->data, $arr);
        }
    }

    public function saveTo(string $path): void
    {
        $driver = $this->drivers->resolve($path);
        $driver->dump($this->data, $path);
    }

    // Interface
    public function get(string $key, mixed $default = null): mixed
    {
        return array_key_exists($key, $this->data) ? $this->data[$key] : $default;
    }

    public function set(string $key, mixed $value): void
    {
        $this->data[$key] = $value;
    }

    // ArrayAccess
    public function offsetExists(mixed $offset): bool
    {
        return array_key_exists((string)$offset, $this->data);
    }

    public function offsetGet(mixed $offset): mixed
    {
        return $this->get((string)$offset);
    }

    public function offsetSet(mixed $offset, mixed $value): void
    {
        $this->set((string)$offset, $value);
    }

    public function offsetUnset(mixed $offset): void
    {
        unset($this->data[(string)$offset]);
    }
}