<?php

namespace AmgGroup\Tests;

use AmgGroup\Config;
use PHPUnit\Framework\TestCase;

/**
 * Test case for the Config class.
 */
class ConfigTest extends TestCase
{
    /**
     * Path to the test fixtures directory.
     */
    private const FIXTURES_DIR = __DIR__ . '/fixtures';

    /**
     * Reset the Config singleton instance before each test.
     */
    protected function setUp(): void
    {
        // Reset the singleton instance using reflection
        $reflectionClass = new \ReflectionClass(Config::class);
        $instanceProperty = $reflectionClass->getProperty('instance');
        $instanceProperty->setAccessible(true);
        $instanceProperty->setValue(null, null);
    }

    /**
     * Test that getInstance returns a singleton instance.
     */
    public function testGetInstanceReturnsSingletonInstance(): void
    {
        $defaultFile = self::FIXTURES_DIR . '/default.yml';
        $localFile = self::FIXTURES_DIR . '/local.yml';

        $config1 = Config::getInstance($defaultFile, $localFile);
        $config2 = Config::getInstance($defaultFile, $localFile);

        $this->assertSame($config1, $config2, 'getInstance should return the same instance');
    }

    /**
     * Test loading configuration from default file only.
     */
    public function testLoadDefaultConfigOnly(): void
    {
        $defaultFile = self::FIXTURES_DIR . '/default.yml';
        $nonExistentFile = self::FIXTURES_DIR . '/nonexistent.yml';

        $config = Config::getInstance($defaultFile, $nonExistentFile);

        $this->assertTrue($config->configIsLoaded(), 'Config should be loaded');
        $this->assertEquals('Test Config Library', $config->get('app.name'), 'Should load app name from default config');
        $this->assertEquals('test', $config->get('app.environment'), 'Should load app environment from default config');
        $this->assertEquals('test_database', $config->get('database.name'), 'Should load database name from default config');
    }

    /**
     * Test loading configuration from both default and local files.
     */
    public function testLoadDefaultAndLocalConfig(): void
    {
        $defaultFile = self::FIXTURES_DIR . '/default.yml';
        $localFile = self::FIXTURES_DIR . '/local.yml';

        $config = Config::getInstance($defaultFile, $localFile);

        $this->assertTrue($config->configIsLoaded(), 'Config should be loaded');
        
        // Values from default config that aren't overridden in local config
        $this->assertEquals('Test Config Library', $config->get('app.name'), 'Should load app name from default config');
        
        // Values overridden in local config
        $this->assertEquals(true, $config->get('app.debug'), 'Should load app debug from local config');
        $this->assertEquals('development', $config->get('app.environment'), 'Should load app environment from local config');
        $this->assertEquals('127.0.0.1', $config->get('database.host'), 'Should load database host from local config');
        $this->assertEquals('local_test_database', $config->get('database.name'), 'Should load database name from local config');
    }

    /**
     * Test getting configuration values with dot notation.
     */
    public function testGetConfigValue(): void
    {
        $defaultFile = self::FIXTURES_DIR . '/default.yml';
        $config = Config::getInstance($defaultFile);

        // Test getting values at different levels
        $this->assertEquals('Test Config Library', $config->get('app.name'), 'Should get nested value with dot notation');
        $this->assertEquals(['name' => 'Test Config Library', 'debug' => false, 'environment' => 'test'], $config->get('app'), 'Should get entire section');
        
        // Test getting non-existent values
        $this->assertNull($config->get('nonexistent.key'), 'Should return null for non-existent key');
        $this->assertEquals('default', $config->get('nonexistent.key', 'default'), 'Should return default value for non-existent key');
    }

    /**
     * Test setting configuration values.
     */
    public function testSetConfigValue(): void
    {
        $defaultFile = self::FIXTURES_DIR . '/default.yml';
        $config = Config::getInstance($defaultFile);

        // Set a new value
        $config->set('app.new_setting', 'new_value');
        $this->assertEquals('new_value', $config->get('app.new_setting'), 'Should set and get new value');

        // Override an existing value
        $config->set('app.name', 'Updated Name');
        $this->assertEquals('Updated Name', $config->get('app.name'), 'Should override existing value');

        // Set a deeply nested value
        $config->set('deep.nested.value', 'nested_value');
        $this->assertEquals('nested_value', $config->get('deep.nested.value'), 'Should set and get deeply nested value');
    }

    /**
     * Test getting all configuration values.
     */
    public function testGetAllConfig(): void
    {
        $defaultFile = self::FIXTURES_DIR . '/default.yml';
        $config = Config::getInstance($defaultFile);

        $allConfig = $config->getAll();
        
        $this->assertIsArray($allConfig, 'getAll should return an array');
        $this->assertArrayHasKey('app', $allConfig, 'Config should have app section');
        $this->assertArrayHasKey('database', $allConfig, 'Config should have database section');
        $this->assertArrayHasKey('env', $allConfig, 'Config should have env section');
    }

    /**
     * Test setting environment variables.
     */
    public function testSetEnvironmentVariables(): void
    {
        $defaultFile = self::FIXTURES_DIR . '/default.yml';
        $config = Config::getInstance($defaultFile);

        // The setEnvironmentVariables method is called automatically when loading the config
        // So we just need to check if the environment variables are set
        $this->assertEquals('test', getenv('APP_ENV'), 'Should set APP_ENV environment variable');
        $this->assertEquals('false', getenv('DEBUG'), 'Should set DEBUG environment variable');
    }

    /**
     * Test error handling when loading an invalid YAML file.
     */
    public function testLoadInvalidYamlFile(): void
    {
        $invalidFile = self::FIXTURES_DIR . '/invalid.yml';
        
        $this->expectException(\Exception::class);
        $this->expectExceptionMessageMatches('/Failed to parse configuration file/');
        
        Config::getInstance($invalidFile);
    }

    /**
     * Test error handling when loading a non-existent file.
     */
    public function testLoadNonExistentFile(): void
    {
        $nonExistentFile = self::FIXTURES_DIR . '/nonexistent.yml';
        
        // This should not throw an exception, but the config should not be loaded
        $config = Config::getInstance($nonExistentFile);
        
        $this->assertFalse($config->configIsLoaded(), 'Config should not be loaded');
        $this->assertEmpty($config->getAll(), 'Config should be empty');
    }
}