<?php

namespace AmgGroup;

use Symfony\Component\Yaml\Yaml;
use Symfony\Component\Yaml\Exception\ParseException;

/**
 * Class Config
 *
 * Singleton class to load configuration settings from default and local YAML files.
 * The local configuration file overrides any matching settings from the default file.
 */
class Config
{
    /**
     * @var bool Set to true once a config file has been loaded
     */
    private bool $configLoaded = false;
    /**
     * Holds the single instance of Config.
     *
     * @var Config|null
     */
    private static ?Config $instance = null;

    /**
     * @var array Configuration data loaded from YAML files.
     */
    private array $config = [];

    private static string $logger = '';

    /**
     * Private constructor to prevent direct instantiation.
     *
     * @param string $defaultFile Path to the default configuration file.
     * @param string $localFile Path to the local configuration file.
     *
     * @throws \Exception If the default configuration file cannot be loaded or parsed.
     */
    private function __construct(string $defaultFile = __DIR__ . '/../../../../config/default.yml', string $localFile = __DIR__ . '/../../../../config/local.yml')
    {

        //$this->logger=Logger::getInstance();
        //$this->logger->debug("Config Default File : {$defaultFile}");
        //$this->logger->debug("Config Local File : {$localFile}");
        // Load default configuration
        $this->loadFile($defaultFile);

        // Override with local configuration if the file exists
        if (file_exists($localFile)) {
            $this->loadFile($localFile, true);
        }
    }

    private static function writeToLogFile($data)
    {
        $timestamp = date('Y-m-d H:i:s');
        $formattedData = print_r($data, true);
        file_put_contents(self::$logger, "[{$timestamp}] {$formattedData}\n", FILE_APPEND);
    }

    /**
     * Provides access to the single instance of Config.
     *
     * @param string $defaultFile Path to the default configuration file.
     * @param string $localFile Path to the local configuration file.
     * @return Config
     * @throws \Exception
     */
    public static function getInstance(
        string $defaultFile = __DIR__ . '/../../config/default.yml',
        string $localFile = __DIR__ . '/../../config/local.yml'
    ): Config
    {
        if (!self::$logger) {
            self::$logger = tempnam(sys_get_temp_dir(), 'amgConfig_');
        }
        self::writeToLogFile("Config Default File : {$defaultFile}");
        self::writeToLogFile("Config Local File : {$localFile}");
        if (self::$instance === null) {
            self::$instance = new self($defaultFile, $localFile);
        }
        return self::$instance;
    }

    /**
     * Load a configuration file and optionally merge it into the current config.
     *
     * @param string $filePath Path to the YAML configuration file.
     * @param bool $override Whether to merge this configuration into the current config.
     *
     * @throws \Exception If the file cannot be loaded or parsed.
     */
    private function loadFile(string $filePath, bool $override = false): void
    {
        if (!file_exists($filePath)) {
            return;
        }
        try {
            $data = Yaml::parseFile($filePath);

            if (!is_array($data)) {
                throw new \Exception("Configuration file {$filePath} does not contain a valid YAML array.");
            }

            $this->config = $override ? array_replace_recursive($this->config, $data) : $data;

        } catch (ParseException $e) {
            throw new \Exception("Failed to parse configuration file {$filePath}: " . $e->getMessage());
        }
        $this->configLoaded = true;
        $this->setEnvironmentVariables();
    }

    /**
     * Get the value of a configuration key.
     *
     * @param string $key The configuration key (supports dot notation, e.g., "debug.level").
     * @param mixed $default A default value to return if the key does not exist.
     *
     * @return mixed The value of the configuration key or the default value.
     */
    public function get(string $key, $default = null)
    {
        $keys = explode('.', $key);
        $value = $this->config;

        foreach ($keys as $keyPart) {
            if (is_array($value) && array_key_exists($keyPart, $value)) {
                $value = $value[$keyPart];
            } else {
                return $default;
            }
        }

        return $value;
    }

    /**
     * Set a configuration value by key.
     *
     * This allows dynamically modifying the configuration values at runtime.
     *
     * @param string $key The configuration key (supports dot notation, e.g., "debug.level").
     * @param mixed $value The value to set for the specified configuration key.
     *
     * @return void
     */
    public function set(string $key, mixed $value): void
    {
        $keys = explode('.', $key);
        $config = &$this->config;

        foreach ($keys as $keyPart) {
            if (!isset($config[$keyPart]) || !is_array($config[$keyPart])) {
                $config[$keyPart] = [];
            }
            $config = &$config[$keyPart];
        }

        $config = $value;
    }

    /**
     * Set environment variables based on the 'env' section of the config.
     *
     * This method loops through all the key-value pairs in the 'env' section
     * and assigns them as environment variables.
     *
     * @return void
     */
    public function setEnvironmentVariables(): void
    {
        // Get the 'env' section of the configuration.
        $envConfig = $this->get('env', []);

        if (!is_array($envConfig)) {
            error_log("The 'env' configuration is not an array. Skipping environment variable setup.");
            return;
        }

        // Set each environment variable
        foreach ($envConfig as $key => $value) {
            // Use putenv to set the environment variable.
            // This will make it available to the current process.
            putenv("{$key}={$value}");

            // Also set it in the $_ENV array for compatibility.
            $_ENV[$key] = $value;

            // Optionally, set it in the $_SERVER array for some server environments.
            $_SERVER[$key] = $value;
        }
    }

    public function configIsLoaded()
    {
        return $this->configLoaded;
    }


    /**
     * Get the entire configuration array.
     *
     * @return array The full configuration data.
     */
    public function getAll(): array
    {
        return $this->config;
    }

    /**
     * Prevent cloning of the instance.
     */
    private function __clone()
    {
    }

    /**
     * Prevent unserialization of the instance.
     */
    public function __wakeup()
    {
    }
}
