<?php
declare(strict_types=1);

namespace AmgGroup\FileCopier;

use Composer\IO\IOInterface;

final class FileCopier
{
    public function __construct(
        private string $rootDir,
        private IOInterface $io
    ) {}

    /**
     * Classic path-based copy (file or directory).
     *
     * @param string         $source     Path relative to project root (or absolute)
     * @param string         $target     Path relative to project root (or absolute)
     * @param bool           $recursive  Copy directories recursively if true
     * @param bool           $overwrite  Overwrite existing files
     * @param string|int|null $fileMode  File chmod (e.g., "0644" or 0644). If null, no chmod is applied to files.
     * @param string|int     $dirMode    Directory chmod (e.g., "0755" or 0755) applied to created directories
     * @param array<string>  $exclude    List of exclude patterns (project-root-relative, forward slashes)
     */
    public function process(
        string $source,
        string $target,
        bool $recursive,
        bool $overwrite,
        string|int|null $fileMode,
        string|int $dirMode,
        array $exclude = []
    ): void {
        $src = $this->resolvePath($source);
        $dst = $this->resolvePath($target);

        if (!file_exists($src)) {
            throw new \RuntimeException(sprintf('Source does not exist: %s', $source));
        }

        if (is_file($src) || is_link($src)) {
            // For file copy, excludes do not apply (only directories are filtered)
            $this->copyFileTask($src, $dst, $overwrite, $fileMode, $dirMode);
            return;
        }

        if (is_dir($src)) {
            $this->copyDirectoryTask($src, $dst, $recursive, $overwrite, $fileMode, $dirMode, $exclude);
            return;
        }

        throw new \RuntimeException(sprintf('Unsupported source type: %s', $source));
    }

    /**
     * Glob-based copy: patterns matched against project-root-relative paths.
     * Target is treated as a directory; matched items are mapped under it
     * preserving structure relative to each pattern's non-wildcard base.
     *
     * @param array<string>  $patterns   Glob patterns (supports ** and {a,b})
     * @param string         $targetDir  Destination directory
     * @param bool           $recursive  If a matched item is a directory: recursive copy if true
     * @param bool           $overwrite
     * @param string|int|null $fileMode
     * @param string|int     $dirMode
     * @param array<string>  $exclude
     */
    public function processGlob(
        array $patterns,
        string $targetDir,
        bool $recursive,
        bool $overwrite,
        string|int|null $fileMode,
        string|int $dirMode,
        array $exclude = []
    ): void {
        $dstDir = $this->resolvePath($targetDir);
        $this->ensureDirectory($dstDir, $dirMode);

        // Normalize and expand braces in patterns
        $normalized = [];
        foreach ($patterns as $p) {
            if (!\is_string($p) || $p === '') {
                continue;
            }
            foreach ($this->expandBraces($this->normalizeSlashes($p)) as $ep) {
                $normalized[] = $ep;
            }
        }
        if ($normalized === []) {
            throw new \RuntimeException('No valid glob patterns provided.');
        }

        // Build list of matches (deduped), mapping each to its base
        $matches = []; // relPath => ['abs' => absPath, 'is_dir' => bool, 'base' => baseRel]
        foreach ($normalized as $pattern) {
            [$baseRel, $regex] = $this->compilePattern($pattern);
            $baseAbs = $this->resolvePath($baseRel);

            // If base doesn't exist, nothing matches for this pattern
            if (!is_dir($baseAbs) && !is_file($baseAbs)) {
                continue;
            }

            // Include the base directory/file itself as a potential match
            $items = [new \SplFileInfo($baseAbs)];

            if (is_dir($baseAbs)) {
                $iter = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator($baseAbs, \FilesystemIterator::SKIP_DOTS),
                    \RecursiveIteratorIterator::SELF_FIRST
                );
                foreach ($iter as $info) {
                    $items[] = $info;
                }
            }

            foreach ($items as $info) {
                /** @var \SplFileInfo $info */
                $abs = $info->getPathname();
                $rel = $this->toRootRelative($abs);
                if ($rel === null) {
                    // Outside root; ignore
                    continue;
                }

                $relNorm = $this->normalizeSlashes($rel);
                if (!preg_match($regex, $relNorm)) {
                    continue;
                }

                if ($this->shouldExclude($relNorm, $exclude)) {
                    // Skip excluded match
                    continue;
                }

                // For directories, include them as matches, too (so we can mirror structure)
                $isDir = $info->isDir();
                $matches[$relNorm] = [
                    'abs'    => $abs,
                    'is_dir' => $isDir,
                    'base'   => $baseRel,
                ];
            }
        }

        if ($matches === []) {
            $this->io->write('[file-copier] No files matched glob patterns.');
            return;
        }

        // Process each match
        foreach ($matches as $rel => $meta) {
            $abs    = $meta['abs'];
            $isDir  = $meta['is_dir'];
            $base   = $meta['base'];

            // Compute relative-to-base subpath for mapping
            $subRel = $this->stripBase($rel, $this->normalizeSlashes($base));
            $dest   = rtrim($dstDir, "/\\") . DIRECTORY_SEPARATOR . $this->fromRootRelative($subRel);

            if ($isDir) {
                if ($recursive) {
                    $this->copyDirectoryTask($abs, $dest, true, $overwrite, $fileMode, $dirMode, $exclude);
                } else {
                    // Shallow: create the directory only
                    $this->ensureDirectory($dest, $dirMode);
                    $this->io->write(sprintf('[file-copier] Created directory (glob, non-recursive): %s', $dest));
                }
            } else {
                // Ensure parent dir exists
                $this->copyFileTask($abs, $dest, $overwrite, $fileMode, $dirMode);
            }
        }
    }

    private function resolvePath(string $path): string
    {
        if ($this->isAbsolutePath($path)) {
            return $this->normalizePath($path);
        }
        return $this->normalizePath($this->rootDir . DIRECTORY_SEPARATOR . $path);
    }

    private function isAbsolutePath(string $path): bool
    {
        if ($path === '') return false;
        if ($path[0] === '/' || $path[0] === '\\') return true; // Unix or UNC
        if (\strlen($path) > 1 && ctype_alpha($path[0]) && $path[1] === ':') return true; // Windows drive
        return false;
    }

    private function normalizePath(string $path): string
    {
        $path = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path);
        $parts = array_filter(explode(DIRECTORY_SEPARATOR, $path), 'strlen');
        $absolutes = [];
        foreach ($parts as $part) {
            if ('.' == $part) continue;
            if ('..' == $part) {
                array_pop($absolutes);
            } else {
                $absolutes[] = $part;
            }
        }
        $res = implode(DIRECTORY_SEPARATOR, $absolutes);
        if ($path !== '' && ($path[0] === '/' || $path[0] === '\\')) {
            $res = DIRECTORY_SEPARATOR . $res;
        } elseif (\strlen($path) > 1 && ctype_alpha($path[0]) && $path[1] === ':') {
            $res = $path[0] . ':' . DIRECTORY_SEPARATOR . $res;
        }
        return $res;
    }

    private function normalizeSlashes(string $path): string
    {
        return str_replace('\\', '/', $path);
    }

    private function fromRootRelative(string $rel): string
    {
        return str_replace('/', DIRECTORY_SEPARATOR, ltrim($rel, '/'));
    }

    private function toRootRelative(string $abs): ?string
    {
        $root = $this->normalizeSlashes($this->normalizePath($this->rootDir));
        $absN = $this->normalizeSlashes($this->normalizePath($abs));
        
        $root = rtrim($root, '/') . '/';
        
        if (str_starts_with($absN, $root)) {
            return substr($absN, \strlen($root));
        }
        if ($absN === rtrim($root, '/')) {
            return '';
        }
        return null;
    }

    private function copyFileTask(
        string $srcFile,
        string $dstPath,
        bool $overwrite,
        string|int|null $fileMode,
        string|int $dirMode
    ): void {
        // If target is a directory or ends with a separator, put file inside using basename
        $targetIsDir = (is_dir($dstPath)) || $this->endsWithSeparator($dstPath);
        $dstFile = $targetIsDir ? rtrim($dstPath, "/\\") . DIRECTORY_SEPARATOR . basename($srcFile) : $dstPath;

        $this->ensureDirectory(dirname($dstFile), $dirMode);

        if (file_exists($dstFile) && !$overwrite) {
            $this->io->write(sprintf('[file-copier] Skipping existing file (overwrite=false): %s', $dstFile));
            return;
        }

        if (!@copy($srcFile, $dstFile)) {
            $err = error_get_last();
            throw new \RuntimeException(sprintf('Failed to copy file "%s" -> "%s"%s',
                $srcFile, $dstFile, $err ? (': ' . $err['message']) : ''
            ));
        }

        $this->io->write(sprintf('[file-copier] Copied file: %s -> %s', $srcFile, $dstFile));

        if ($fileMode !== null) {
            $this->applyMode($dstFile, $fileMode, false);
        }
    }

    private function copyDirectoryTask(
        string $srcDir,
        string $dstPath,
        bool $recursive,
        bool $overwrite,
        string|int|null $fileMode,
        string|int $dirMode,
        array $exclude = []
    ): void {
        $dstDir = $dstPath;
        $this->ensureDirectory($dstDir, $dirMode);

        $iterator = new \DirectoryIterator($srcDir);
        foreach ($iterator as $info) {
            if ($info->isDot()) {
                continue;
            }

            $src = $info->getPathname();
            $dst = $dstDir . DIRECTORY_SEPARATOR . $info->getBasename();

            $rel = $this->toRootRelative($src);
            $rel = $rel ? $this->normalizeSlashes($rel) : null;

            if ($rel !== null && $this->shouldExclude($rel, $exclude)) {
                // Skip excluded file/dir entirely
                $this->io->write(sprintf('[file-copier] Excluded: %s', $rel));
                continue;
            }

            if ($info->isDir()) {
                if ($recursive) {
                    $this->copyDirectoryTask($src, $dst, true, $overwrite, $fileMode, $dirMode, $exclude);
                } else {
                    $this->ensureDirectory($dst, $dirMode);
                    $this->io->write(sprintf('[file-copier] Created directory (non-recursive): %s', $dst));
                }
            } else {
                $this->copyFileTask($src, $dst, $overwrite, $fileMode, $dirMode);
            }
        }
    }

    private function endsWithSeparator(string $path): bool
    {
        $last = substr($path, -1);
        return $last === '/' || $last === '\\';
    }

    private function ensureDirectory(string $dir, string|int $dirMode): void
    {
        if (is_dir($dir)) {
            return;
        }

        if (!@mkdir($dir, $this->parseMode($dirMode), true) && !is_dir($dir)) {
            $err = error_get_last();
            throw new \RuntimeException(sprintf('Failed to create directory "%s"%s',
                $dir, $err ? (': ' . $err['message']) : ''
            ));
        }

        // Explicitly chmod after creation to ensure exact mode (umask may have applied)
        $this->applyMode($dir, $dirMode, true);
    }

    private function applyMode(string $path, string|int $mode, bool $isDir): void
    {
        $m = $this->parseMode($mode);
        if (\DIRECTORY_SEPARATOR === '\\') {
            // On Windows, chmod is best-effort and often a no-op
            @chmod($path, $m);
            return;
        }
        if (!@chmod($path, $m)) {
            $this->io->write(sprintf('[file-copier] Warning: chmod failed for %s (%s)', $isDir ? 'directory' : 'file', $path));
        }
    }

    private function parseMode(string|int $mode): int
    {
        if (\is_int($mode)) {
            return $mode;
        }
        $mode = trim($mode);
        if (str_starts_with($mode, '0o')) {
            return octdec(substr($mode, 2));
        }
        if ($mode === '' || !ctype_digit(ltrim($mode, '0'))) {
            throw new \InvalidArgumentException(sprintf('Invalid chmod mode: %s', $mode));
        }
        return octdec($mode);
    }

    /**
     * Expand simple brace patterns like:
     *   a/{b,c}/d -> a/b/d, a/c/d
     * Supports nested braces by recursive expansion.
     *
     * @return array<string>
     */
    private function expandBraces(string $pattern): array
    {
        $results = [];

        $pos = strpos($pattern, '{');
        if ($pos === false) {
            return [$pattern];
        }

        $depth = 0;
        $start = $pos;
        $end   = null;
        for ($i = $pos; $i < strlen($pattern); $i++) {
            if ($pattern[$i] === '{') $depth++;
            if ($pattern[$i] === '}') $depth--;
            if ($depth === 0) {
                $end = $i;
                break;
            }
        }
        if ($end === null) {
            // Unbalanced braces; return as-is
            return [$pattern];
        }

        $prefix = substr($pattern, 0, $start);
        $inside = substr($pattern, $start + 1, $end - $start - 1);
        $suffix = substr($pattern, $end + 1);

        $parts = explode(',', $inside);
        foreach ($parts as $part) {
            foreach ($this->expandBraces($prefix . $part . $suffix) as $expanded) {
                $results[] = $expanded;
            }
        }

        return $results;
    }

    /**
     * Compile a glob pattern into (baseRel, regex).
     * baseRel is the non-wildcard leading path (relative to root), used for mapping & traversal.
     *
     * Supports **, *, ?, [], and brace-expanded input.
     *
     * @return array{0:string,1:string} baseRel, regex (delimitered)
     */
    private function compilePattern(string $pattern): array
    {
        $pattern = ltrim($pattern, '/'); // make it root-relative in our matching scheme

        // Find index of first wildcard metachar
        $wildPos = $this->firstWildcardPos($pattern);
        $baseRel = $wildPos === null
            ? $pattern
            : rtrim(substr($pattern, 0, $wildPos), '/');

        // Build regex
        $re = '';
        $i = 0;
        $len = strlen($pattern);
        while ($i < $len) {
            $ch = $pattern[$i];

            if ($ch === '*') {
                if ($i + 1 < $len && $pattern[$i + 1] === '*') {
                    // '**'
                    $i += 2;
                    if ($i < $len && $pattern[$i] === '/') {
                        // '**/...'
                        $re .= '(.*\/)?';
                        $i++;
                    } elseif ($i === $len && $i >= 2 && ($i == 2 || $pattern[$i - 3] === '/')) {
                        // '.../**' or '**'
                        $re .= '(.*)?';
                    } else {
                        $re .= '.*';
                    }
                    continue;
                }
                $re .= '[^/]*';
                $i++;
                continue;
            }

            if ($ch === '?') {
                $re .= '[^/]';
                $i++;
                continue;
            }

            if ($ch === '[') {
                $end = strpos($pattern, ']', $i + 1);
                if ($end === false) {
                    $re .= '\[';
                    $i++;
                    continue;
                }
                $re .= substr($pattern, $i, $end - $i + 1);
                $i = $end + 1;
                continue;
            }

            if (preg_match('~[.\\+^$(){}=!<>|:-]~', $ch)) {
                $re .= '\\' . $ch;
            } else {
                $re .= $ch;
            }
            $i++;
        }

        $regex = '~^' . $re . '$~u';

        return [$baseRel, $regex];
    }

    private function firstWildcardPos(string $pattern): ?int
    {
        $chars = ['*', '?', '[', '{'];
        $min = null;
        foreach ($chars as $c) {
            $pos = strpos($pattern, $c);
            if ($pos !== false) {
                $min = $min === null ? $pos : min($min, $pos);
            }
        }
        // Also treat '**' as wildcard
        $pos2 = strpos($pattern, '**');
        if ($pos2 !== false) {
            $min = $min === null ? $pos2 : min($min, $pos2);
        }
        return $min;
    }

    private function stripBase(string $relPath, string $baseRel): string
    {
        $relPath = ltrim($this->normalizeSlashes($relPath), '/');
        $baseRel = trim($this->normalizeSlashes($baseRel), '/');

        if ($baseRel === '') {
            return $relPath;
        }
        if ($relPath === $baseRel) {
            return basename($relPath);
        }
        if (str_starts_with($relPath, $baseRel . '/')) {
            return substr($relPath, strlen($baseRel) + 1);
        }
        // If not under base (edge case), return full relative path
        return $relPath;
    }

    private function shouldExclude(string $relPath, array $exclude): bool
    {
        if ($exclude === []) return false;
        $relPath = ltrim($this->normalizeSlashes($relPath), '/');
        foreach ($exclude as $pat) {
            if (!\is_string($pat) || $pat === '') continue;
            foreach ($this->expandBraces($this->normalizeSlashes($pat)) as $ep) {
                [, $regex] = $this->compilePattern($ep);
                if (preg_match($regex, $relPath) === 1) {
                    return true;
                }
                
                // Also check if any parent directory is excluded
                $parts = explode('/', $relPath);
                if (count($parts) > 1) {
                    $prefix = '';
                    for ($i = 0; $i < count($parts) - 1; $i++) {
                        $prefix = ($prefix === '') ? $parts[$i] : $prefix . '/' . $parts[$i];
                        if (preg_match($regex, $prefix) === 1 || preg_match($regex, $prefix . '/') === 1) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }
}
