<?php
declare(strict_types=1);

namespace AmgGroup\FileCopier\Tests;

use AmgGroup\FileCopier\FileCopier;
use Composer\IO\IOInterface;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Filesystem\Filesystem;

final class FileCopierTest extends TestCase
{
    private string $tempDir;
    private Filesystem $fs;
    private IOInterface $io;

    protected function setUp(): void
    {
        $this->tempDir = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'file_copier_tests_' . uniqid();
        $this->fs = new Filesystem();
        $this->fs->mkdir($this->tempDir);
        
        $this->io = $this->createMock(IOInterface::class);
    }

    protected function tearDown(): void
    {
        $this->fs->remove($this->tempDir);
    }

    private function createFiles(array $files): void
    {
        foreach ($files as $path => $content) {
            $fullPath = $this->tempDir . DIRECTORY_SEPARATOR . str_replace('/', DIRECTORY_SEPARATOR, $path);
            $this->fs->mkdir(dirname($fullPath));
            file_put_contents($fullPath, $content);
        }
    }

    public function testSimpleFileCopy(): void
    {
        $this->createFiles(['source/app.ini' => 'foo=bar']);
        $copier = new FileCopier($this->tempDir, $this->io);

        $copier->process('source/app.ini', 'target/app.ini', false, true, '0644', '0755', []);

        $this->assertFileExists($this->tempDir . '/target/app.ini');
        $this->assertEquals('foo=bar', file_get_contents($this->tempDir . '/target/app.ini'));
    }

    public function testDirectoryRecursiveCopy(): void
    {
        $this->createFiles([
            'resources/assets/js/app.js' => 'console.log("hi");',
            'resources/assets/css/style.css' => 'body { color: red; }',
            'resources/assets/js/app.js.map' => '{}',
            'resources/assets/drafts/old.txt' => 'old',
        ]);

        $copier = new FileCopier($this->tempDir, $this->io);

        $exclude = [
            '**/*.map',
            '**/drafts/**'
        ];

        $copier->process('resources/assets', 'public/assets', true, true, '0644', '0755', $exclude);

        $this->assertFileExists($this->tempDir . '/public/assets/js/app.js');
        $this->assertFileExists($this->tempDir . '/public/assets/css/style.css');
        $this->assertFileDoesNotExist($this->tempDir . '/public/assets/js/app.js.map');
        $this->assertDirectoryDoesNotExist($this->tempDir . '/public/assets/drafts');
    }

    public function testGlobCopy(): void
    {
        $this->createFiles([
            'resources/img/logo.png' => 'logo',
            'resources/img/icons/add.png' => 'add',
            'resources/img/icons/remove.jpg' => 'remove',
        ]);

        $copier = new FileCopier($this->tempDir, $this->io);

        $copier->processGlob(['resources/img/**/*.png'], 'public/img', true, true, '0644', '0755', []);

        $this->assertFileExists($this->tempDir . '/public/img/logo.png');
        $this->assertFileExists($this->tempDir . '/public/img/icons/add.png');
        $this->assertFileDoesNotExist($this->tempDir . '/public/img/icons/remove.jpg');
    }

    public function testMultipleGlobsWithExcludes(): void
    {
        $this->createFiles([
            'resources/snippets/test.txt' => 'test',
            'resources/snippets/WIP/secret.txt' => 'secret',
            'resources/docs/readme.md' => 'readme',
            'resources/docs/todo.draft.md' => 'todo',
        ]);

        $copier = new FileCopier($this->tempDir, $this->io);

        $patterns = [
            'resources/snippets/**/*.{txt,md}',
            'resources/docs/**/*.md'
        ];
        $exclude = [
            '**/WIP/**',
            '**/*.draft.md'
        ];

        $copier->processGlob($patterns, 'public/docs', false, true, '0644', '0755', $exclude);

        $this->assertFileExists($this->tempDir . '/public/docs/test.txt');
        $this->assertFileExists($this->tempDir . '/public/docs/readme.md');
        $this->assertFileDoesNotExist($this->tempDir . '/public/docs/WIP/secret.txt');
        $this->assertFileDoesNotExist($this->tempDir . '/public/docs/todo.draft.md');
    }

    public function testDirectoryMatchRecursiveFalse(): void
    {
        $this->createFiles([
            'resources/docs/info.txt' => 'info',
            'resources/docs/sub/more.txt' => 'more',
        ]);

        $copier = new FileCopier($this->tempDir, $this->io);

        // Pattern matches the directory 'resources/docs'
        $copier->processGlob(['resources/docs'], 'public/test', false, true, '0644', '0755', []);

        $this->assertDirectoryExists($this->tempDir . '/public/test/docs');
        $this->assertFileDoesNotExist($this->tempDir . '/public/test/docs/info.txt');
    }
}
