<?php
declare(strict_types=1);

namespace AmgGroup\FileCopier;

use Composer\Composer;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Plugin\PluginInterface;
use Composer\Script\Event;
use Composer\Script\ScriptEvents;

final class FileCopierPlugin implements PluginInterface, EventSubscriberInterface
{
    private Composer $composer;
    private IOInterface $io;

    public function activate(Composer $composer, IOInterface $io): void
    {
        $this->composer = $composer;
        $this->io       = $io;
    }

    public function deactivate(Composer $composer, IOInterface $io): void {}
    public function uninstall(Composer $composer, IOInterface $io): void {}

    public static function getSubscribedEvents(): array
    {
        return [
            ScriptEvents::POST_INSTALL_CMD => 'runFromEvent',
            ScriptEvents::POST_UPDATE_CMD  => 'runFromEvent',
        ];
    }

    public static function run(Event $event): void
    {
        $io       = $event->getIO();
        $composer = $event->getComposer();

        (new self())->execute($composer, $io);
    }

    public function runFromEvent(Event $event): void
    {
        $this->execute($event->getComposer(), $event->getIO());
    }

    private function execute(Composer $composer, IOInterface $io): void
    {
        $rootPackage = $composer->getPackage();
        $extra       = $rootPackage->getExtra();

        if (!isset($extra['file-copier'])) {
            $io->write('<info>[file-copier]</info> No "file-copier" configuration found in root composer.json.');
            return;
        }

        $config = $extra['file-copier'];
        $copies = $config['copies'] ?? [];

        if (!\is_array($copies) || $copies === []) {
            $io->write('<info>[file-copier]</info> No copies to process.');
            return;
        }

        $failOnError = (bool)($config['failOnError'] ?? false);

        // Compute project root reliably from vendor-dir
        $vendorDir = $composer->getConfig()->get('vendor-dir');
        $rootDir   = \realpath($vendorDir . DIRECTORY_SEPARATOR . '..') ?: \getcwd();

        $copier = new FileCopier($rootDir, $io);

        $io->write('<info>[file-copier]</info> Processing copy tasks...');

        $errors = 0;

        foreach ($copies as $idx => $task) {
            $taskNo = $idx + 1;

            if (!\is_array($task)) {
                $io->writeError(\sprintf('[file-copier] Task #%d is not a valid object/array.', $taskNo));
                $errors++;
                if ($failOnError) {
                    throw new \RuntimeException('Invalid file-copier task configuration.');
                }
                continue;
            }

            $source    = $task['source']   ?? null;
            $glob      = $task['glob']     ?? null;  // NEW
            $target    = $task['target']   ?? null;
            $recursive = (bool)($task['recursive'] ?? false);
            $overwrite = (bool)($task['overwrite'] ?? true);

            $fileMode  = $task['mode']    ?? null;    // for files
            $dirMode   = $task['dirMode'] ?? '0755';  // for directories

            $exclude   = $task['exclude'] ?? [];      // NEW
            if (\is_string($exclude)) {
                $exclude = [$exclude];
            }
            if (!\is_array($exclude)) {
                $exclude = [];
            }

            if (!\is_string($target) || $target === '') {
                $io->writeError(\sprintf('[file-copier] Task #%d missing a valid "target".', $taskNo));
                $errors++;
                if ($failOnError) {
                    throw new \RuntimeException('Missing target in file-copier task.');
                }
                continue;
            }

            try {
                if ($glob !== null) {
                    // glob can be string or array
                    $patterns = \is_string($glob) ? [$glob] : (is_array($glob) ? $glob : []);
                    if ($patterns === []) {
                        throw new \RuntimeException('Empty "glob" in file-copier task.');
                    }
                    $copier->processGlob($patterns, $target, $recursive, $overwrite, $fileMode, $dirMode, $exclude);
                } else {
                    if (!\is_string($source) || $source === '') {
                        throw new \RuntimeException('Task must have a non-empty "source" when "glob" is not provided.');
                    }
                    $copier->process($source, $target, $recursive, $overwrite, $fileMode, $dirMode, $exclude);
                }
            } catch (\Throwable $e) {
                $errors++;
                $io->writeError(\sprintf('[file-copier] Task #%d failed: %s', $taskNo, $e->getMessage()));
                if ($failOnError) {
                    throw $e;
                }
            }
        }

        if ($errors > 0) {
            $io->writeError(\sprintf('<error>[file-copier]</error> Completed with %d error(s).', $errors));
        } else {
            $io->write('<info>[file-copier]</info> All copy tasks completed successfully.');
        }
    }
}
