<?php

declare(strict_types=1);

namespace AmgGroup\Tests;

// Ensure PSR-16 interface is available for the test environment
require_once __DIR__ . '/../src/Compat/SimpleCachePolyfill.php';

use AmgGroup\ApiClient;
use AmgGroup\ConfigInterface;
use AmgGroup\TokenProviderInterface;
use GuzzleHttp\ClientInterface as GuzzleClient;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

final class ApiClientCacheTest extends TestCase
{
    private function makeConfigStub(
        bool $empty = true,
        ?string $baseUri = null,
        array $defaultHeaders = [],
        int $defaultTtl = 300,
        int $negativeTtl = 30
    ): ConfigInterface {
        return new class($empty, $baseUri, $defaultHeaders, $defaultTtl, $negativeTtl) implements ConfigInterface {
            public function __construct(
                private bool $empty,
                private ?string $baseUri,
                private array $defaultHeaders,
                private int $defaultTtl,
                private int $negativeTtl
            ) {}
            public function get(string $key, mixed $default = null): mixed {
                return match ($key) {
                    'api.base_uri', 'api.baseUrl', 'api.baseURL', 'base_uri' => $this->baseUri ?? $default,
                    'api.default_headers', 'api.defaultHeaders', 'default_headers' => $this->defaultHeaders ?: $default,
                    'api.cache.default_ttl', 'api.default_ttl', 'cache.default_ttl', 'default_ttl' => $this->defaultTtl ?? $default,
                    'api.cache.negative_ttl', 'api.negative_ttl', 'cache.negative_ttl', 'negative_ttl' => $this->negativeTtl ?? $default,
                    default => $default,
                };
            }
            public function set(string $key, mixed $value): void {}
            public function isEmpty(): bool { return $this->empty; }
            public function getLoadedPaths(): array { return []; }
            public function getSkippedPaths(): array { return []; }
        };
    }

    private function makeCache(): CacheInterface
    {
        return new class implements CacheInterface {
            private array $store = [];
            public function get($key, $default = null): mixed { return $this->store[$key] ?? $default; }
            public function set($key, $value, $ttl = null): bool { $this->store[$key] = $value; return true; }
            public function delete($key): bool { unset($this->store[$key]); return true; }
            public function clear(): bool { $this->store = []; return true; }
            public function getMultiple($keys, $default = null): iterable { $out=[]; foreach ($keys as $k) { $out[$k]=$this->get($k,$default);} return $out; }
            public function setMultiple($values, $ttl = null): bool { foreach ($values as $k=>$v) { $this->set($k,$v,$ttl);} return true; }
            public function deleteMultiple($keys): bool { foreach ($keys as $k) { $this->delete($k);} return true; }
            public function has($key): bool { return array_key_exists($key, $this->store); }
        };
    }

    private function makeApiClient(
        GuzzleClient $http,
        TokenProviderInterface $tokenProvider,
        LoggerInterface $logger,
        ?string $baseUri = null,
        array $defaultHeaders = []
    ): ApiClient {
        $headers = ['Accept' => 'application/json'] + $defaultHeaders;
        $config = $this->makeConfigStub(false, $baseUri, $headers);
        return new ApiClient($http, $logger, $tokenProvider, $config);
    }

    public function testGetIsCachedAndSecondCallAvoidsHttp(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('abc');

        $response = new Response(200, ['Content-Type' => 'application/json'], '{"ok":true}');
        $http->expects($this->once())
            ->method('request')
            ->with('GET', 'https://api.example.com/v1/users', $this->anything())
            ->willReturn($response);

        $client = $this->makeApiClient($http, $tokenProvider, $logger, 'https://api.example.com/v1', []);
        $client->withCache($this->makeCache());

        $r1 = $client->request('GET', '/users');
        $this->assertSame(200, $r1->getStatusCode());

        // Second call should be served from cache without hitting HTTP
        $r2 = $client->request('GET', '/users');
        $this->assertSame(200, $r2->getStatusCode());
    }

    public function testWriteInvalidatesRelatedKeys(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('abc');

        $cache = $this->makeCache();

        // First GET populates cache
        $http->expects($this->exactly(2))
            ->method('request')
            ->willReturnOnConsecutiveCalls(
                new Response(200, [], '[]'),
                new Response(204, [], '')
            );

        $client = $this->makeApiClient($http, $tokenProvider, $logger, 'https://api.example.com/v1', []);
        $client->withCache($cache);
        $client->request('GET', '/users'); // cached
        $client->request('POST', '/users', ['json' => ['x' => 1]]); // should purge

        // After invalidation, next GET should hit HTTP again (we don't set expectation here; new mock simpler)
        $http2 = $this->createMock(GuzzleClient::class);
        $http2->expects($this->once())
            ->method('request')
            ->willReturn(new Response(200, [], '[]'));
        $client2 = $this->makeApiClient($http2, $tokenProvider, $logger, 'https://api.example.com/v1', []);
        $client2->withCache($cache);
        $client2->request('GET', '/users');
    }

    public function testEtagFlow304ServesCachedBody(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('abc');
        $cache = $this->makeCache();

        $first = new Response(200, ['ETag' => 'W/"123"'], '{"n":1}');
        $second = new Response(304, [], null);

        $http->expects($this->exactly(2))
            ->method('request')
            ->willReturnOnConsecutiveCalls($first, $second);

        $client = $this->makeApiClient($http, $tokenProvider, $logger, 'https://api.example.com/v1', []);
        $client->withCache($cache);
        $r1 = $client->request('GET', '/users');
        $this->assertSame(200, $r1->getStatusCode());
        $r2 = $client->request('GET', '/users');
        $this->assertSame(200, $r2->getStatusCode());
        $this->assertSame('{"n":1}', (string)$r2->getBody());
    }

    public function testNegativeCaching404(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('abc');
        $cache = $this->makeCache();

        $http->expects($this->once())
            ->method('request')
            ->willReturn(new Response(404, [], ''));

        $client = $this->makeApiClient($http, $tokenProvider, $logger, 'https://api.example.com/v1', []);
        $client->withCache($cache);
        $r1 = $client->request('GET', '/missing');
        $this->assertSame(404, $r1->getStatusCode());
        // Second call should be served from negative cache without HTTP
        $r2 = $client->request('GET', '/missing');
        $this->assertSame(404, $r2->getStatusCode());
    }

    public function testGetAllPagesAggregatedCaching(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('abc');
        $cache = $this->makeCache();

        $page = new Response(200, [], json_encode(['value' => [['id' => 1]]], JSON_THROW_ON_ERROR));
        $http->expects($this->once())
            ->method('request')
            ->willReturn($page);

        $client = $this->makeApiClient($http, $tokenProvider, $logger, 'https://api.example.com/v1', []);
        $client->withCache($cache);
        $first = $client->getAllPages('/users');
        $this->assertSame([[ 'id' => 1 ]], $first);
        $second = $client->getAllPages('/users');
        $this->assertSame([[ 'id' => 1 ]], $second);
    }
}
