<?php

namespace AmgGroup;

use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\ResponseInterface;

/**
 * DryRunApiClient
 * ---------------
 * Decorator that wraps ApiClient and short-circuits destructive methods when dry-run is enabled.
 */
final class DryRunApiClient
{
    /** @var string[] */
    private array $destructiveMethods = ['PATCH', 'DELETE', 'CREATE', 'POST', 'PUT'];

    private ?ResponseInterface $lastResponse = null;
    private int $statusCode = 0;

    public function __construct(
        private readonly ApiClient $inner,
        private bool $dry = true,
        array $destructiveMethods = []
    ) {
        if ($destructiveMethods) {
            $this->destructiveMethods = array_map('strtoupper', $destructiveMethods);
        }
    }

    public function setDry(bool $dry): self
    {
        $this->dry = $dry;
        return $this;
    }

    public function setDefaultHeaders(array $headers): self
    {
        $this->inner->setDefaultHeaders($headers);
        return $this;
    }

    public function getAllPages(string $url, array $params = [], string $nextKey = '@odata.nextLink'): array
    {
        return $this->inner->getAllPages($url, $params, $nextKey);
    }

    /**
     * Proxy request with dry-run protection for destructive methods.
     */
    public function request(string $method, string $url, array $options = []): ?ResponseInterface
    {
        $upper = strtoupper($method);
        if ($this->dry && in_array($upper, $this->destructiveMethods, true)) {
            // Return a synthetic success response
            $this->lastResponse = new Response(200, ['X-Dry-Run' => 'true'], '');
            $this->statusCode = 200;
            return $this->lastResponse;
        }
        $res = $this->inner->request($method, $url, $options);
        $this->lastResponse = $res;
        $this->statusCode = $this->inner->getStatusCode();
        return $res;
    }

    public function getLastResponse(): ?ResponseInterface
    {
        return $this->lastResponse ?? $this->inner->getLastResponse();
    }

    public function getStatusCode(): int
    {
        return $this->statusCode ?: $this->inner->getStatusCode();
    }
}
