<?php

namespace AmgGroup;

use Psr\Http\Message\ResponseInterface;

/**
 * ResourceClient
 * --------------
 * A small convenience wrapper that binds an ApiClient to a specific resource path
 * and exposes CRUD-style helpers. Keeps the transport clean while offering
 * ergonomic resource operations.
 */
final class ResourceClient
{
    public function __construct(
        private readonly ApiClient $api,
        private readonly string $resourcePath
    ) {
    }

    /**
     * List all items under the resource path, with optional reindex by a field (default: 'id').
     */
    public function list(string $reIndex = 'id'): array
    {
        $data = $this->api->getAllPages($this->resourcePath);
        if ($reIndex) {
            $indexed = [];
            foreach ($data as $item) {
                if (is_array($item) && array_key_exists($reIndex, $item)) {
                    $indexed[$item[$reIndex]] = $item;
                }
            }
            return $indexed;
        }
        return $data;
    }

    /** Get a single item by id; unwrap single-element arrays for convenience. */
    public function get(mixed $id): mixed
    {
        $data = $this->api->getAllPages(rtrim($this->resourcePath, '/') . '/' . $id);
        if (is_array($data) && count($data) === 1) {
            return $data[0];
        }
        return $data;
    }

    public function post(mixed $data): ?ResponseInterface
    {
        return $this->api->request('POST', $this->resourcePath, ['json' => $data]);
    }

    public function create(mixed $data): ?ResponseInterface
    {
        return $this->api->request('CREATE', $this->resourcePath, ['json' => $data]);
    }

    public function put(mixed $id, mixed $data): ?ResponseInterface
    {
        return $this->api->request('PUT', rtrim($this->resourcePath, '/') . '/' . $id, ['json' => $data]);
    }

    public function patch(mixed $id, mixed $data): ?ResponseInterface
    {
        return $this->api->request('PATCH', rtrim($this->resourcePath, '/') . '/' . $id, ['json' => $data]);
    }

    public function delete(mixed $id): ?ResponseInterface
    {
        return $this->api->request('DELETE', rtrim($this->resourcePath, '/') . '/' . $id);
    }
}
