<?php

namespace AmgGroup;

use GuzzleHttp\ClientInterface as GuzzleClient;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;


/**
 * Class ApiClient
 * ----------------
 * A generic API client with pagination support.
 */
final class ApiClient
{
    /** @var string|null Base URI to prefix relative URLs */
    private ?string $baseUri;

    /** Default headers to include on every request (merged with provided options) */
    private array $defaultHeaders;

    /** Optional PSR-16 cache for GET requests */
    private ?CacheInterface $cache = null;

    /** Default TTLs (in seconds) */
    private int $defaultTtl = 300;
    private int $negativeTtl = 30;

    /** Public last response/status for backwards compatibility */
    public ResponseInterface|null $return = null;
    public int $statusCode = 0;

    public function __construct(
        private readonly GuzzleClient $http,
        private readonly LoggerInterface $logger,
        private readonly TokenProviderInterface $tokenProvider,
        private readonly ConfigInterface $config,
        ?CacheInterface $cache = null,
    ) {
        $this->cache = $cache;
        $this->logger->info('ApiClient constructed');
        if ($this->config->isEmpty()) {
            // Keep the caller-visible behavior per user's note
            $this->logger->error('No config loaded');
            throw new \Exception('Config not loaded');
        }
        // Read common key variants with sensible defaults
        $this->baseUri = $this->getFirstConfig([
            'base_uri', 'api.base_uri', 'api.baseUrl', 'api.baseURL'
        ], null);
        $this->defaultHeaders = $this->getFirstConfig([
            'default_headers', 'api.default_headers', 'api.defaultHeaders'
        ], []);
        $this->defaultTtl = (int)$this->getFirstConfig([
            'default_ttl', 'api.default_ttl', 'api.cache.default_ttl', 'cache.default_ttl'
        ], 300);
        $this->negativeTtl = (int)$this->getFirstConfig([
            'negative_ttl', 'api.negative_ttl', 'api.cache.negative_ttl', 'cache.negative_ttl'
        ], 30);
    }


    // Singleton removed to support dependency injection

    // Authentication responsibility moved to TokenProviderInterface

    /**
     * Perform a GET request with pagination support.
     *
     * @param string $url Base API endpoint.
     * @param array $params Optional query parameters.
     * @param string $nextKey JSON key path to next URL (default: 'next').
     * @return array All combined response data pages.
     */
    public function getAllPages(string $url, array $params = [], string $nextKey = '@odata.nextLink'): array
    {
        $results = [];

        // Attempt aggregated cache if enabled
        $aggCacheKey = null;
        if ($this->cache) {
            $initialUrl = $url . (empty($params) ? '' : ('?' . http_build_query($params)));
            $aggCacheKey = $this->buildCacheKey('GET', $this->normalizeAbsoluteUrl($initialUrl), [], true);
            try {
                $cached = $this->cache->get($aggCacheKey);
                if (is_array($cached) && array_key_exists('agg', $cached)) {
                    return $cached['agg'];
                }
            } catch (\Throwable $e) {
                // ignore cache errors
            }
        }

        try {
            $fullUrl = $url . (empty($params) ? '' : ('?' . http_build_query($params)));
            //echo "Full URL: " . $fullUrl . PHP_EOL;
            while ($fullUrl) {
                $response = $this->request('GET', $fullUrl);
                $data = json_decode($response->getBody(), true);

                // Append results from the current page
                if (isset($data['value']) && is_array($data['value'])) {
                    $results = array_merge($results, $data['value']);
                }

                // Determine next URL (if pagination is used)
                $fullUrl = $data[$nextKey] ?? null;
            }
        } catch (RequestException $e) {
            echo "Request failed: " . $e->getMessage();
        }

        // Store aggregated cache if enabled
        if ($this->cache && $aggCacheKey) {
            try {
                $this->cache->set($aggCacheKey, ['agg' => $results], $this->defaultTtl);
                $this->indexKeyForPrefix($aggCacheKey, $this->extractRootPrefixFromUrl($url));
            } catch (\Throwable $e) {
                // ignore cache errors
            }
        }

        return $results;
    }

    /**
     * Send a generic HTTP request.
     *
     * @param string $method HTTP method (GET, POST, etc).
     * @param string $url Endpoint URL.
     * @param array $options Guzzle options (headers, json, query, etc).
     * @return ResponseInterface|null
     * @throws GuzzleException
     */
    public function request(string $method, string $url, array $options = []): ?ResponseInterface
    {
        // Normalize method
        if (strtoupper($method) === 'CREATE') {
            $method = 'POST';
        }

        // Apply base URI for relative URLs only; keep absolute URLs as-is
        $finalUrl = $this->normalizeAbsoluteUrl($url);

        // Merge headers and add Authorization
        $headers = $options['headers'] ?? [];
        $headers = array_merge([
            'Authorization' => 'Bearer ' . $this->tokenProvider->getAccessToken(),
            'Accept' => 'application/json',
        ], $this->defaultHeaders, $headers);
        $options['headers'] = $headers;

        $isGet = strtoupper($method) === 'GET';
        $noCache = (bool)($options['no_cache'] ?? false);
        $ttl = (int)($options['ttl'] ?? $this->defaultTtl);
        $negativeTtl = (int)($options['negative_ttl'] ?? $this->negativeTtl);

        $cacheKey = null;
        $etag = null;
        $cachedPayload = null;

        if ($isGet && $this->cache && !$noCache) {
            $varyHeaders = $this->extractVaryHeaders($headers);
            $cacheKey = $this->buildCacheKey('GET', $finalUrl, $varyHeaders);
            try {
                $cachedPayload = $this->cache->get($cacheKey);
                if (is_array($cachedPayload)) {
                    // If negative cached, serve directly without HTTP
                    if (!empty($cachedPayload['negative'])) {
                        $resp = new Response(404, [], '');
                        $this->return = $resp;
                        $this->statusCode = 404;
                        return $this->return;
                    }
                    // If we have payload without ETag, serve from cache without HTTP
                    if (empty($cachedPayload['etag']) && isset($cachedPayload['status'])) {
                        $resp = new Response(
                            $cachedPayload['status'] ?? 200,
                            $cachedPayload['headers'] ?? [],
                            $cachedPayload['body'] ?? ''
                        );
                        $this->return = $resp;
                        $this->statusCode = $resp->getStatusCode();
                        return $this->return;
                    }
                    // Else if we have an ETag, do conditional request
                    if (isset($cachedPayload['etag'])) {
                        $etag = $cachedPayload['etag'];
                        if ($etag) {
                            $options['headers']['If-None-Match'] = $etag;
                        }
                    }
                }
            } catch (\Throwable $e) {
                // ignore cache errors
            }
        }

        try {
            $this->return = $this->http->request($method, $finalUrl, $options);
        } catch (RequestException $e) {
            $this->logger->error('Request error', [
                'message' => $e->getMessage(),
                'method' => $method,
                'url' => $finalUrl,
                'options' => $options,
                'response' => $e->hasResponse() ? (string)$e->getResponse()->getBody() : null,
            ]);
            $this->return = $e->getResponse();
        }
        $this->statusCode = $this->return ? $this->return->getStatusCode() : 0;

        // Handle caching behaviors
        if ($isGet && $this->cache && !$noCache) {
            if ($this->return) {
                $status = $this->return->getStatusCode();
                if ($status === 304 && is_array($cachedPayload)) {
                    // Serve from cache
                    $resp = new Response(
                        $cachedPayload['status'] ?? 200,
                        $cachedPayload['headers'] ?? [],
                        $cachedPayload['body'] ?? ''
                    );
                    $this->return = $resp;
                    $this->statusCode = $resp->getStatusCode();
                    return $this->return;
                }
                if ($status >= 200 && $status < 300) {
                    // Cache successful GET
                    $body = (string)$this->return->getBody();
                    $headersOut = $this->headersToArray($this->return->getHeaders());
                    $etagResp = $this->return->getHeaderLine('ETag');
                    $payload = [
                        'status' => $status,
                        'headers' => $headersOut,
                        'body' => $body,
                        'etag' => $etagResp ?: null,
                    ];
                    try {
                        $this->cache->set($cacheKey, $payload, $ttl);
                        $this->indexKeyForPrefix($cacheKey, $this->extractRootPrefixFromUrl($finalUrl));
                    } catch (\Throwable $e) {
                        // ignore cache errors
                    }
                } elseif ($status === 404) {
                    // Negative caching for 404
                    try {
                        $this->cache->set($cacheKey, ['status' => 404, 'body' => '', 'headers' => [], 'negative' => true], $negativeTtl);
                        $this->indexKeyForPrefix($cacheKey, $this->extractRootPrefixFromUrl($finalUrl));
                    } catch (\Throwable $e) {
                        // ignore cache errors
                    }
                }
            } else {
                // Network error with no response: if we had a cached success, serve it
                if (is_array($cachedPayload) && isset($cachedPayload['status'])) {
                    $resp = new Response(
                        $cachedPayload['status'] ?? 200,
                        $cachedPayload['headers'] ?? [],
                        $cachedPayload['body'] ?? ''
                    );
                    $this->return = $resp;
                    $this->statusCode = $resp->getStatusCode();
                }
            }
        } elseif (!$isGet && $this->cache) {
            // Invalidate on writes
            $prefix = $this->extractRootPrefixFromUrl($finalUrl);
            $this->purgeCacheFor($prefix);
        }

        return $this->return;
    }

    /**
     * Set or override default headers for subsequent requests (merged with per-call headers).
     * Returns $this to allow fluent style.
     */
    public function setDefaultHeaders(array $headers): self
    {
        $this->defaultHeaders = $headers;
        return $this;
    }

    /** Attach or replace the cache instance (mutable, returns $this for fluency). */
    public function withCache(?CacheInterface $cache): self
    {
        $this->cache = $cache;
        return $this;
    }

    /** Get the last HTTP response returned by the client (or null if none/failed). */
    public function getLastResponse(): ?ResponseInterface
    {
        return $this->return;
    }

    /** Get the HTTP status code of the last request (0 if none/failed). */
    public function getStatusCode(): int
    {
        return $this->statusCode;
    }

    /** Purge all cached keys under the given path prefix (e.g., "/users"). */
    public function purgeCacheFor(string $pathPrefix): void
    {
        if (!$this->cache) {
            return;
        }
        try {
            $base = $this->baseTag();
            $idxKey = "graph:{$base}:prefix:" . trim($pathPrefix, '/');
            $keys = $this->cache->get($idxKey);
            if (is_array($keys)) {
                foreach ($keys as $k) {
                    $this->cache->delete($k);
                }
            }
            // Clear the index itself
            $this->cache->delete($idxKey);
        } catch (\Throwable $e) {
            // ignore cache errors
        }
    }

    // ===== Helpers =====

    /** Get the first non-null config value from the provided keys, else default. */
    private function getFirstConfig(array $keys, mixed $default = null): mixed
    {
        foreach ($keys as $k) {
            $val = $this->config->get($k, null);
            if ($val !== null) {
                return $val;
            }
        }
        return $default;
    }

    /** Normalize to absolute URL using baseUri if needed. */
    private function normalizeAbsoluteUrl(string $url): string
    {
        $isAbsolute = (bool)parse_url($url, PHP_URL_SCHEME);
        if ($isAbsolute) {
            return $url;
        }
        return $this->baseUri ? rtrim($this->baseUri, '/') . '/' . ltrim($url, '/') : $url;
    }

    /** Extract vary headers relevant for representation. Authorization must NOT be included. */
    private function extractVaryHeaders(array $headers): array
    {
        $vary = [];
        $candidates = ['ConsistencyLevel'];
        foreach ($candidates as $h) {
            if (isset($headers[$h])) {
                $vary[$h] = $headers[$h];
            }
        }
        return $vary;
    }

    /** Build the cache key. */
    private function buildCacheKey(string $method, string $absoluteUrl, array $varyHeaders = [], bool $aggregated = false): string
    {
        $base = $this->baseTag();
        $parts = parse_url($absoluteUrl);
        $path = $parts['path'] ?? '';
        $query = $parts['query'] ?? '';
        parse_str($query, $qArr);
        ksort($qArr);
        $sortedQuery = http_build_query($qArr);
        $varyHash = '';
        if (!empty($varyHeaders)) {
            ksort($varyHeaders);
            $varyHash = substr(sha1(json_encode($varyHeaders)), 0, 8);
        }
        $type = $aggregated ? 'agg' : 'res';
        return sprintf('graph:%s:%s:%s:%s?%s&h=%s', $base, $type, strtoupper($method), trim($path, '/'), $sortedQuery, $varyHash);
    }

    /** Convert headers map (array<string, string[]>) to simple array<string, string>. */
    private function headersToArray(array $headers): array
    {
        $flat = [];
        foreach ($headers as $k => $vals) {
            $flat[$k] = is_array($vals) ? implode(', ', $vals) : (string)$vals;
        }
        return $flat;
    }

    /** Determine base tag (host + api version if present). */
    private function baseTag(): string
    {
        $base = $this->baseUri ?: '';
        if (!$base) {
            return 'local';
        }
        $parts = parse_url($base);
        $host = $parts['host'] ?? 'local';
        $path = trim($parts['path'] ?? '', '/');
        $ver = '';
        if ($path !== '') {
            $segs = explode('/', $path);
            $first = $segs[0] ?? '';
            if (preg_match('/^v\d+/i', $first)) {
                $ver = '/' . $first;
            }
        }
        return $host . $ver;
    }

    /** Extract the root prefix from a URL path (e.g., "/users"). */
    private function extractRootPrefixFromUrl(string $absoluteUrl): string
    {
        $parts = parse_url($absoluteUrl);
        $path = trim($parts['path'] ?? '', '/');
        $segs = explode('/', $path);
        // Skip api version segment like v1
        if (isset($segs[0]) && preg_match('/^v\d+/i', $segs[0])) {
            array_shift($segs);
        }
        $root = $segs[0] ?? '';
        return '/' . $root;
    }

    /** Track key under a prefix-specific index for invalidation. */
    private function indexKeyForPrefix(string $key, string $pathPrefix): void
    {
        if (!$this->cache) return;
        try {
            $idxKey = "graph:" . $this->baseTag() . ":prefix:" . trim($pathPrefix, '/');
            $set = $this->cache->get($idxKey);
            $arr = is_array($set) ? $set : [];
            if (!in_array($key, $arr, true)) {
                $arr[] = $key;
                $this->cache->set($idxKey, $arr, $this->defaultTtl);
            }
        } catch (\Throwable $e) {
            // ignore cache errors
        }
    }
}
