### ApiClient — Universal API client wrapper

`AmgGroup\ApiClient` is a small wrapper around an injected HTTP client that adds:

- Automatic bearer token injection via a pluggable `TokenProviderInterface`.
- Optional base URI handling for relative endpoints.
- Simple pagination helper tailored for Graph-style `@odata.nextLink`.
- PSR-3 logging hooks.

This class is DI-friendly: no singletons, no internal client creation, no hard-coded authentication.

#### Constructor
```php
public function __construct(
    GuzzleHttp\ClientInterface $http,
    Psr\Log\LoggerInterface $logger,
    AmgGroup\TokenProviderInterface $tokenProvider,
    AmgGroup\ConfigInterface $config,
    ?string $baseUri = null,
    array $defaultHeaders = [],
    ?Psr\SimpleCache\CacheInterface $cache = null,
    int $defaultTtl = 300,
    int $negativeTtl = 30
)
```

- `$http`: an already-configured Guzzle client (or adapter implementing `ClientInterface`).
- `$logger`: PSR-3 logger.
- `$tokenProvider`: supplies bearer tokens. Implement `TokenProviderInterface` for your auth system.
- `$config`: application config (used for early sanity checks in this package).
- `$baseUri`: optional base URI to prefix relative endpoints.
- `$defaultHeaders`: headers merged into every request.
- `$cache`: optional PSR-16 cache to enable transparent GET caching (null disables caching).
- `$defaultTtl`: default TTL (seconds) for successful GET responses.
- `$negativeTtl`: default TTL (seconds) for negative caching (e.g., 404).

#### Making requests
```php
$response = $apiClient->request('GET', '/users', [
    'query' => ['$top' => 10]
]);

if ($response) {
    $data = json_decode((string)$response->getBody(), true);
}
```

- `request(string $method, string $url, array $options = [])` forwards to the injected HTTP client after:
  - Resolving `$url` against `$baseUri` (if set)
  - Adding `Authorization: Bearer <token>` and `Accept: application/json`
  - Merging `$defaultHeaders` and any `$options['headers']`

Note: A convenience alias of the non-standard method `CREATE` maps to `POST` for backwards compatibility.

#### Pagination helper
```php
$all = $apiClient->getAllPages('/users', ['$top' => 50]);
```

- Iteratively follows `@odata.nextLink` by default and merges `value` arrays from each page.
- For other APIs, pass a different `$nextKey`.

### Caching

This client supports optional PSR-16 caching for GET requests. Caching is fully disabled unless you pass a cache instance.

Enable caching via constructor or by cloning with `withCache()`:

```php
use Symfony\Component\Cache\Adapter\ArrayAdapter;
use Symfony\Component\Cache\Psr16Cache;

$psr16 = new Psr16Cache(new ArrayAdapter());

// Option A: via constructor
$api = new AmgGroup\ApiClient($http, $logger, $tokenProvider, $config, baseUri: 'https://graph.microsoft.com/v1.0', cache: $psr16, defaultTtl: 300, negativeTtl: 30);

// Option B: clone with cache later
$api = $api->withCache($psr16);
```

Behavior and knobs:
- Only successful 2xx GET responses are cached by default for `defaultTtl` seconds.
- Negative caching: 404 responses are cached briefly (default `negativeTtl` seconds).
- ETag revalidation: if a response contains an `ETag`, it is stored and subsequent GETs include `If-None-Match`. A `304 Not Modified` serves the cached body and headers, while a `200` with a new `ETag` refreshes the cache.
- Per-call overrides: pass `ttl` and/or `negative_ttl` in `$options` to `request('GET', ...)`.
- Per-call opt-out: pass `no_cache => true` in `$options` to skip cache for that call.
- Invalidation on writes: `POST`, `PUT`, `PATCH`, `DELETE` automatically purge cached keys under the same top-level resource (e.g., `/users`, `/groups`). You can also call `purgeCacheFor('/users')` explicitly.
- `getAllPages()` automatically benefits since it uses `request('GET', ...)` under the hood. In addition, it caches the final aggregated array for the initial URL so repeated pagination walks are avoided during the TTL window.

Cache key format:
```
graph:{base}:{type}:{METHOD}:{url_path}?{sorted_query}&h={vary_hash}
```
Where:
- `{base}` distinguishes tenants/environments (host + optional API version from `baseUri`).
- `{type}` is `res` for single responses and `agg` for aggregated `getAllPages()` results.
- `vary_hash` considers a small set of representation-affecting headers like `ConsistencyLevel`. Authorization is never part of the key.

Laravel example:
```php
// In a service provider
$psr16 = app('cache')->store()->getStore(); // Many Laravel stores already implement PSR-16 via adapters
$api = new AmgGroup\ApiClient($http, $logger, $tokenProvider, $config, baseUri: config('services.graph.base_uri'), cache: $psr16);
```

Symfony example:
```php
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Psr16Cache;

$psr16 = new Psr16Cache(new FilesystemAdapter(namespace: 'graph', defaultLifetime: 0));
$api = new AmgGroup\ApiClient($http, $logger, $tokenProvider, $config, baseUri: 'https://graph.microsoft.com/v1.0', cache: $psr16);
```

Explicit invalidation:
```php
// After a bulk write to /users related resources
$api->purgeCacheFor('/users');
```

#### TokenProviderInterface
Provide your own implementation that returns a bearer token string:
```php
final class MyTokenProvider implements AmgGroup\TokenProviderInterface {
    public function getAccessToken(): string { /* fetch or refresh */ }
}
```

For Microsoft Graph client-credentials flow, implement a provider that POSTs to the tenant token endpoint and caches `access_token` until expiry.

#### Example wiring
```php
$http = new GuzzleHttp\Client(['timeout' => 10]);
$tokenProvider = new MyTokenProvider(/* ... */);
$api = new AmgGroup\ApiClient(
    http: $http,
    logger: $logger,
    tokenProvider: $tokenProvider,
    config: $config,
    baseUri: 'https://graph.microsoft.com/v1.0'
);

$users = $api->getAllPages('/users', ['select' => 'id,displayName']);
```

#### Notes
- This class no longer owns authentication; keep that in your `TokenProviderInterface` implementation.
- Replace any legacy singleton usage with constructor injection via your container.
- Errors are logged via PSR-3; where applicable, `request()` returns `null` on transport errors.
