<?php

declare(strict_types=1);

namespace AmgGroup\Tests;

use AmgGroup\ApiClient;
use AmgGroup\ConfigInterface;
use AmgGroup\TokenProviderInterface;
use GuzzleHttp\ClientInterface as GuzzleClient;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;
use Psr\Http\Message\ResponseInterface;

final class ApiClientTest extends TestCase
{
    private function makeConfigStub(bool $empty = true): ConfigInterface
    {
        // Implement the known methods and add isEmpty() used by ApiClient
        return new class($empty) implements ConfigInterface {
            public function __construct(private bool $empty) {}
            public function get(string $key, mixed $default = null): mixed { return $default; }
            public function set(string $key, mixed $value): void {}
            public function isEmpty(): bool { return $this->empty; }
        };
    }

    private function makeApiClient(
        GuzzleClient $http,
        TokenProviderInterface $tokenProvider,
        LoggerInterface $logger,
        ?string $baseUri = null,
        array $defaultHeaders = []
    ): ApiClient {
        $config = $this->makeConfigStub(true); // must be true to avoid constructor exception in current implementation
        return new ApiClient($http, $logger, $tokenProvider, $config, $baseUri, $defaultHeaders);
    }

    public function testRequestAddsAuthorizationDefaultHeadersAndBaseUriAndMapsCreateToPost(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('abc123');

        $expectedResponse = new Response(200, ['Content-Type' => 'application/json'], '{}');

        $http->expects($this->once())
            ->method('request')
            ->with(
                // CREATE must map to POST
                $this->equalTo('POST'),
                // baseUri + relative path
                $this->equalTo('https://api.example.com/v1/users'),
                // headers merged and Authorization added
                $this->callback(function (array $options) {
                    $this->assertArrayHasKey('headers', $options);
                    $headers = $options['headers'];
                    $this->assertSame('Bearer abc123', $headers['Authorization']);
                    $this->assertSame('application/json', $headers['Accept']);
                    // default header present and can be overridden by options
                    $this->assertSame('foo', $headers['X-Default']);
                    $this->assertSame('bar', $headers['X-Override']);
                    return true;
                })
            )
            ->willReturn($expectedResponse);

        $client = $this->makeApiClient(
            http: $http,
            tokenProvider: $tokenProvider,
            logger: $logger,
            baseUri: 'https://api.example.com/v1',
            defaultHeaders: ['X-Default' => 'foo', 'X-Override' => 'will-be-overridden']
        );

        $response = $client->request('CREATE', '/users', [
            'headers' => [
                'X-Override' => 'bar'
            ]
        ]);

        $this->assertInstanceOf(ResponseInterface::class, $response);
        $this->assertSame(200, $client->statusCode);
        $this->assertSame($expectedResponse, $client->return);
    }

    public function testRequestHandlesExceptionReturnsNullAndStatusZero(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('abc123');

        $http->expects($this->once())
            ->method('request')
            ->willThrowException(new RequestException('boom', new Request('GET', 'https://api.example.com/fail')));

        $client = $this->makeApiClient($http, $tokenProvider, $logger, 'https://api.example.com');

        $response = $client->request('GET', '/fail');
        $this->assertNull($response);
        $this->assertSame(0, $client->statusCode);
        $this->assertNull($client->return);
    }

    public function testGetAllPagesFollowsNextLinksAndMergesValues(): void
    {
        $http = $this->createMock(GuzzleClient::class);
        $logger = $this->createMock(LoggerInterface::class);
        $tokenProvider = $this->createMock(TokenProviderInterface::class);
        $tokenProvider->method('getAccessToken')->willReturn('token');

        $page1 = new Response(200, [], json_encode([
            'value' => [['id' => 1], ['id' => 2]],
            '@odata.nextLink' => 'https://api.example.com/users?page=2',
        ], JSON_THROW_ON_ERROR));
        $page2 = new Response(200, [], json_encode([
            'value' => [['id' => 3]],
        ], JSON_THROW_ON_ERROR));

        $http->expects($this->exactly(2))
            ->method('request')
            ->willReturnCallback((function() use ($page1, $page2) {
                $call = 0;
                return function(string $method, string $url, array $options) use (&$call, $page1, $page2) {
                    if ($call === 0) {
                        $this->assertSame('GET', $method);
                        $this->assertSame('https://api.example.com/users?top=2', $url);
                        $call++;
                        return $page1;
                    }
                    $this->assertSame('GET', $method);
                    $this->assertSame('https://api.example.com/users?page=2', $url);
                    $call++;
                    return $page2;
                };
            })());

        $client = $this->makeApiClient($http, $tokenProvider, $logger, 'https://api.example.com');

        $all = $client->getAllPages('/users', ['top' => 2]);
        $this->assertCount(3, $all);
        $this->assertSame([['id' => 1], ['id' => 2], ['id' => 3]], $all);
    }
}
