<?php

namespace AmgGroup;

use GuzzleHttp\ClientInterface as GuzzleClient;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;


/**
 * Class ApiClient
 * ----------------
 * A generic API client with pagination support.
 */
final class ApiClient
{
    /** @var string|null Base URI to prefix relative URLs */
    private ?string $baseUri;

    /** Default headers to include on every request (merged with provided options) */
    private array $defaultHeaders;

    /** Public last response/status for backwards compatibility */
    public ResponseInterface|null $return = null;
    public int $statusCode = 0;

    public function __construct(
        private readonly GuzzleClient $http,
        private readonly LoggerInterface $logger,
        private readonly TokenProviderInterface $tokenProvider,
        private readonly ConfigInterface $config,
        ?string $baseUri = null,
        array $defaultHeaders = []
    ) {
        $this->baseUri = $baseUri;
        $this->defaultHeaders = $defaultHeaders;
        $this->logger->info('ApiClient constructed');
        if (!$this->config->isEmpty()) {
            // Keep the caller-visible behavior per user's note
            $this->logger->error('No config loaded');
            throw new \Exception('Config not loaded');
        }
    }


    // Singleton removed to support dependency injection

    // Authentication responsibility moved to TokenProviderInterface

    /**
     * Perform a GET request with pagination support.
     *
     * @param string $url Base API endpoint.
     * @param array $params Optional query parameters.
     * @param string $nextKey JSON key path to next URL (default: 'next').
     * @return array All combined response data pages.
     */
    public function getAllPages(string $url, array $params = [], string $nextKey = '@odata.nextLink'): array
    {
        $results = [];

        try {
            $fullUrl = $url . (empty($params) ? '' : ('?' . http_build_query($params)));
            //echo "Full URL: " . $fullUrl . PHP_EOL;
            while ($fullUrl) {
                $response = $this->request('GET', $fullUrl);
                $data = json_decode($response->getBody(), true);

                // Append results from the current page
                if (isset($data['value']) && is_array($data['value'])) {
                    $results = array_merge($results, $data['value']);
                }

                // Determine next URL (if pagination is used)
                $fullUrl = $data[$nextKey] ?? null;
            }
        } catch (RequestException $e) {
            echo "Request failed: " . $e->getMessage();
        }

        return $results;
    }

    /**
     * Send a generic HTTP request.
     *
     * @param string $method HTTP method (GET, POST, etc).
     * @param string $url Endpoint URL.
     * @param array $options Guzzle options (headers, json, query, etc).
     * @return ResponseInterface|null
     * @throws GuzzleException
     */
    public function request(string $method, string $url, array $options = []): ?ResponseInterface
    {
        // Normalize method
        if (strtoupper($method) === 'CREATE') {
            $method = 'POST';
        }

        // Apply base URI for relative URLs only; keep absolute URLs as-is
        $isAbsolute = (bool)parse_url($url, PHP_URL_SCHEME);
        if ($isAbsolute) {
            $finalUrl = $url;
        } else {
            $finalUrl = $this->baseUri ? rtrim($this->baseUri, '/') . '/' . ltrim($url, '/') : $url;
        }

        // Merge headers and add Authorization
        $headers = $options['headers'] ?? [];
        $headers = array_merge([
            'Authorization' => 'Bearer ' . $this->tokenProvider->getAccessToken(),
            'Accept' => 'application/json',
        ], $this->defaultHeaders, $headers);
        $options['headers'] = $headers;

        try {
            $this->return = $this->http->request($method, $finalUrl, $options);
        } catch (RequestException $e) {
            $this->logger->error('Request error', [
                'message' => $e->getMessage(),
                'method' => $method,
                'url' => $finalUrl,
                'options' => $options,
                'response' => $e->hasResponse() ? (string)$e->getResponse()->getBody() : null,
            ]);
            $this->return = null;
        }
        $this->statusCode = $this->return ? $this->return->getStatusCode() : 0;
        return $this->return;
    }

    /**
     * Set or override default headers for subsequent requests (merged with per-call headers).
     * Returns $this to allow fluent style.
     */
    public function setDefaultHeaders(array $headers): self
    {
        $this->defaultHeaders = $headers;
        return $this;
    }

    /** Get the last HTTP response returned by the client (or null if none/failed). */
    public function getLastResponse(): ?ResponseInterface
    {
        return $this->return;
    }

    /** Get the HTTP status code of the last request (0 if none/failed). */
    public function getStatusCode(): int
    {
        return $this->statusCode;
    }
}
