### ApiClient — Universal API client wrapper

`AmgGroup\ApiClient` is a small wrapper around an injected HTTP client that adds:

- Automatic bearer token injection via a pluggable `TokenProviderInterface`.
- Optional base URI handling for relative endpoints.
- Simple pagination helper tailored for Graph-style `@odata.nextLink`.
- PSR-3 logging hooks.

This class is DI-friendly: no singletons, no internal client creation, no hard-coded authentication.

#### Constructor
```php
public function __construct(
    GuzzleHttp\ClientInterface $http,
    Psr\Log\LoggerInterface $logger,
    AmgGroup\TokenProviderInterface $tokenProvider,
    AmgGroup\ConfigInterface $config,
    ?string $baseUri = null,
    array $defaultHeaders = []
)
```

- `$http`: an already-configured Guzzle client (or adapter implementing `ClientInterface`).
- `$logger`: PSR-3 logger.
- `$tokenProvider`: supplies bearer tokens. Implement `TokenProviderInterface` for your auth system.
- `$config`: application config (used for early sanity checks in this package).
- `$baseUri`: optional base URI to prefix relative endpoints.
- `$defaultHeaders`: headers merged into every request.

#### Making requests
```php
$response = $apiClient->request('GET', '/users', [
    'query' => ['$top' => 10]
]);

if ($response) {
    $data = json_decode((string)$response->getBody(), true);
}
```

- `request(string $method, string $url, array $options = [])` forwards to the injected HTTP client after:
  - Resolving `$url` against `$baseUri` (if set)
  - Adding `Authorization: Bearer <token>` and `Accept: application/json`
  - Merging `$defaultHeaders` and any `$options['headers']`

Note: A convenience alias of the non-standard method `CREATE` maps to `POST` for backwards compatibility.

#### Pagination helper
```php
$all = $apiClient->getAllPages('/users', ['$top' => 50]);
```

- Iteratively follows `@odata.nextLink` by default and merges `value` arrays from each page.
- For other APIs, pass a different `$nextKey`.

#### TokenProviderInterface
Provide your own implementation that returns a bearer token string:
```php
final class MyTokenProvider implements AmgGroup\TokenProviderInterface {
    public function getAccessToken(): string { /* fetch or refresh */ }
}
```

For Microsoft Graph client-credentials flow, implement a provider that POSTs to the tenant token endpoint and caches `access_token` until expiry.

#### Example wiring
```php
$http = new GuzzleHttp\Client(['timeout' => 10]);
$tokenProvider = new MyTokenProvider(/* ... */);
$api = new AmgGroup\ApiClient(
    http: $http,
    logger: $logger,
    tokenProvider: $tokenProvider,
    config: $config,
    baseUri: 'https://graph.microsoft.com/v1.0'
);

$users = $api->getAllPages('/users', ['select' => 'id,displayName']);
```

#### Notes
- This class no longer owns authentication; keep that in your `TokenProviderInterface` implementation.
- Replace any legacy singleton usage with constructor injection via your container.
- Errors are logged via PSR-3; where applicable, `request()` returns `null` on transport errors.
