<?php

namespace AmgGroup;

include_once 'vendor/autoload.php';

use \GuzzleHttp\Client;
use \GuzzleHttp\Exception\GuzzleException;
use \GuzzleHttp\Exception\RequestException;
use \Psr\Http\Message\ResponseInterface;

/**
 * Api Client Class
 * 
 * This class provides a flexible API client implementation for making HTTP requests to various endpoints.
 * It implements a multiton pattern (similar to singleton but with multiple instances based on endpoints)
 * to manage different API endpoint connections efficiently.
 * 
 * Features:
 * - OAuth2 authentication with token management
 * - Support for standard HTTP methods (GET, POST, PUT, PATCH, DELETE)
 * - Pagination handling for large result sets
 * - "Dry run" mode to prevent destructive operations during testing
 * - Logging of API operations
 * 
 * Usage example:
 * ```php
 * // Set authentication parameters
 * Api::setAuthURL('https://api.example.com/oauth/token');
 * Api::setClientID('your-client-id');
 * Api::setClientSecret('your-client-secret');
 * Api::setGrantType('client_credentials');
 * Api::setScope('read write');
 * 
 * // Get an instance for a specific endpoint
 * $api = Api::getInstance('users');
 * 
 * // Disable dry run mode if you want to perform write operations
 * $api->setDry(false);
 * 
 * // Perform API operations
 * $users = $api->list();
 * $user = $api->get(123);
 * $response = $api->post(['name' => 'New User']);
 * ```
 * 
 * @package AmgGroup
 */
class ApiClient
{
    /**
     * @var array|string[] List of API methods that are classed
     * as Destructive, this is used when 'Dry' run is enabled
     * to prevent writing data back to the API while testing.
     */
    private array $destructiveMethods = [
        'PATCH',
        'DELETE',
        'CREATE',
        'POST'
    ];
    /**
     * @var bool When true, do NOT execute 'destructiveMethods', just log them and return success.
     *          This is useful for testing API interactions without making actual changes.
     */
    private bool $dry = true;
    /**
     * @var array Array holding 'Multiton' pattern instances
     */
    private static array $instances = [];

    /**
     * @var Client
     */
    private Client $client;
    /**
     * @var ResponseInterface|null Used to save return data from the last client request
     */
    private ResponseInterface|null $return;
    /**
     * @var int HTTP status code returned from the last client request
     */
    private int $statusCode;
    /**
     * @var string|null Endpoint used for reading/writing from/to the current API
     */
    private ?string $endpoint;
    /**
     * @var Logger|object Logger used for all logging operations.
     */
    private Logger $logger;
    /**
     * @var Config Config class used for reading API credentials and some other settings
     */
    private Config $config;
    /**
     * @var string|null Access token for connecting to the API
     */
    private ?string $accessToken = null;

    /**
     * @var string Authentication URL for OAuth2 token endpoint
     */
    private static string $authURL;

    /**
     * @var string Client ID for OAuth2 authentication
     */
    private static string $clientID;

    /**
     * @var string Client secret for OAuth2 authentication
     */
    private static string $clientSecret;

    /**
     * @var string Grant type for OAuth2 authentication (e.g., 'client_credentials')
     */
    private static string $grantType;

    /**
     * @var string OAuth2 scope defining the access permissions
     */
    private static string $scope;


    /**
     * Private constructor to enforce the use of getInstance() method (Multiton pattern)
     * 
     * Initializes the API client for a specific endpoint, loads configuration,
     * authenticates with the API, and sets up the HTTP client with the access token.
     * 
     * @param string $endpoint The API endpoint to connect to
     * @throws \Exception If configuration is not loaded
     */
    private function __construct(string $endpoint,$clientID, $clientSecret, $grantType, $scope, $authURL)
    {
        $this->endpoint = $endpoint;
        $this->config = Config::getInstance();
        $this->logger = Logger::getInstance();
        $this->logger->info->apiClient("ApiClient Instantiated for endpoint : $endpoint");

        if (!$this->config->configIsLoaded()) {
            $this->logger->error->apiClient("Config not loaded");
            throw new \Exception("Config not loaded");
        }
        $this->logger->info->apiClient('Tenant ID : ' . $this->config->get('graph.tenant_id'));
        $this->authenticate( $clientID, $clientSecret, $grantType, $scope, $authURL);

        $this->client = new Client([
            'headers' => [
                'Authorization' => 'Bearer ' . $this->accessToken,
                'Accept' => 'application/json',
            ]
        ]);
    }

    /**
     * Get an instance of the API client for a specific endpoint
     *
     * This method implements the Multiton pattern, which ensures that only one
     * instance of the API client exists for each unique endpoint. If an instance
     * for the specified endpoint already exists, it returns that instance;
     * otherwise, it creates a new one.
     *
     * If optional fields are not passed, then the system will use a common
     * (i.e., same data across all instances) set of credentials etc.
     * These are set via static methods.
     *
     * @param string $endpoint The API endpoint to connect to
     * @param string|null $clientID
     * @param string|null $clientSecret
     * @param string|null $grantType
     * @param string|null $scope
     * @param string|null $authURL
     * @return self The API client instance for the specified endpoint
     * @throws \Exception If configuration is not loaded (thrown by constructor)
     */
    public static function getInstance(
        string $endpoint,
        ?string $clientID = null,
        ?string $clientSecret = null,
        ?string $grantType = null,
        ?string $scope = null,
        ?string $authURL = null
    ): self
    {
        if (!isset(self::$instances[$endpoint])) {
            self::$instances[$endpoint] = new self($endpoint,$clientID, $clientSecret, $grantType, $scope, $authURL);
        }
        return self::$instances[$endpoint];
    }

    /**
     * Authenticate and store the access token.
     * MSGraph authURL : "https://login.microsoftonline.com/{$this->config->get('graph.tenant_id')}/oauth2/v2.0/token"
     * Uses default field names for the form_params data to be passed to the authentication URL
     * These can be overridden in the config file with the following :
     * api:
     *      clientIDKey:        client_id
     *      clientSecretKey:    client_secret
     *      grantTypeKey:       grant_type
     *      scopeKey:           scope
     *
     *
     *
     * @param string|null $clientID
     * @param string|null $clientSecret
     * @param string|null $grantType
     * @param string|null $scope
     * @param string|null $authURL
     * @throws GuzzleException
     *
     */
    private function authenticate(
        ?string $clientID = null,
        ?string $clientSecret = null,
        ?string $grantType = null,
        ?string $scope = null,
        ?string $authURL = null): void
    {
        $clientID = $clientID ?? self::$clientID;
        $clientSecret = $clientSecret ?? self::$clientSecret;
        $grantType = $grantType ?? self::$grantType;
        $scope = $scope ?? self::$scope;
        $authURL = $authURL ?? self::$authURL;
        $clientIDKey = $this->config->get('api.clientIDKey') ?? 'client_id';
        $clientSecretKey = $this->config->get('api.clientSecretKey') ?? 'client_secret';
        $grantTypeKey = $this->config->get('api.grantTypeKey') ?? 'grant_type';
        $scopeKey = $this->config->get('api.scopeKey') ?? 'scope';
        $client = new Client();
        try {
            $response = $client->post($authURL, [
                'form_params' => [
                    ($clientIDKey !== null ? [$clientIDKey => $clientID] : []),
                    ($clientSecretKey !== null ? [$clientSecretKey => $clientSecret] : []),
                    ($grantTypeKey !== null ? [$grantTypeKey => $grantType] : []),
                    ($scopeKey !== null ? [$scopeKey => $scope] : [])
                ]
            ]);

            // Check the response body
            $body = json_decode($response->getBody()->getContents(), true); // Decode JSON as array
            $accessToken = $body['access_token']; // Extract the 'access_token'
            $this->accessToken = $accessToken;
            return;
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            echo 'Error: ' . $e->getMessage();
            if ($e->hasResponse()) {
                $body = $e->getResponse()->getBody();
                echo 'Response Body: ' . $body;
            }
            return;
        }
    }

    /**
     * Get a list of all records from the current endpoint
     *
     * This method retrieves all records from the API endpoint, handling pagination
     * automatically. It can optionally reindex the results by a specified field.
     *
     * @param string $reIndex Field to use as the array key in the returned results.
     *                        If set (default is 'id'), the returned array will be
     *                        indexed by this field from each record. Set to empty
     *                        string or null to keep the original numeric indexing.
     * @return array Associative array of records, optionally indexed by the specified field
     */
    public function list(string $reIndex = 'id'): array
    {
        $data = $this->getAllPages($this->endpoint);
        if ($reIndex) {
            $newData = [];
            foreach ($data as $key => $datum) {
                $newData[$datum[$reIndex]] = $datum;
            }
            return $newData;
        } else {
            return $data;
        }
    }

    /**
     * Get a single record by its ID
     * 
     * Retrieves a specific record from the API endpoint using its unique identifier.
     * If the API returns an array with a single item (which is common in some APIs),
     * this method will automatically extract that item from the array.
     *
     * @param mixed $id The unique identifier of the record to retrieve
     * @return array|mixed The record data, either as an array or as the raw response
     *                    depending on the API's response format
     */
    public function get($id)
    {
        $data = $this->getAllPages($this->endpoint . '/' . $id);
        if (is_array($data) && count($data) === 1) {
            return $data[0];
        } else {
            return $data;
        }
    }

    /**
     * Post data to the API endpoint
     *
     * Sends a POST request to the current endpoint with the provided data.
     * This is typically used for creating new resources or submitting form data.
     * 
     * Note: In dry run mode, this operation will be logged but not executed.
     *
     * @param mixed $data The data to send in the request body (will be JSON-encoded)
     * @return ResponseInterface|null The HTTP response or null if the request failed
     */
    public function post($data)
    {
        return $this->request('POST', $this->endpoint, ['json' => $data]);
    }

    /**
     * Update a record using PUT method
     *
     * Sends a PUT request to update a specific record identified by its ID.
     * PUT typically replaces the entire resource with the provided data.
     * 
     * Note: In dry run mode, this operation will be logged but not executed.
     *
     * @param mixed $id The unique identifier of the record to update
     * @param mixed $data The new data for the record (will be JSON-encoded)
     * @return ResponseInterface|null The HTTP response or null if the request failed
     */
    public function put($id, $data)
    {
        return $this->request('PUT', $this->endpoint . '/' . $id, ['json' => $data]);
    }

    /**
     * Delete a record
     *
     * Sends a DELETE request to remove a specific record identified by its ID.
     * 
     * Note: In dry run mode, this operation will be logged but not executed.
     *
     * @param mixed $id The unique identifier of the record to delete
     * @return ResponseInterface|null The HTTP response or null if the request failed
     */
    public function delete($id)
    {
        return $this->request('DELETE', $this->endpoint . '/' . $id);
    }

    /**
     * Partially update a record using PATCH method
     *
     * Sends a PATCH request to update specific fields of a record identified by its ID.
     * Unlike PUT, PATCH only updates the fields provided in the data parameter,
     * leaving other fields unchanged.
     * 
     * Note: In dry run mode, this operation will be logged but not executed.
     * Important: Ensure only the fields that need to be updated are included in the data.
     *
     * @param mixed $id The unique identifier of the record to update
     * @param mixed $data The partial data to update (will be JSON-encoded)
     * @return ResponseInterface|null The HTTP response or null if the request failed
     */
    public function patch($id, $data)
    {
        return $this->request('PATCH', $this->endpoint . '/' . $id, ['json' => $data]);
    }

    /**
     * Create a new record
     *
     * This is a semantic alias for the post() method that makes the intention clearer.
     * It uses a special 'CREATE' method type that is mapped to POST in the request() method.
     * 
     * Note: In dry run mode, this operation will be logged but not executed.
     *
     * @param mixed $data The data for the new record (will be JSON-encoded)
     * @return ResponseInterface|null The HTTP response or null if the request failed
     */
    public function create($data)
    {
        return $this->request('CREATE', $this->endpoint, ['json' => $data]);
    }


    /**
     * Perform a GET request with automatic pagination support
     *
     * This method handles API endpoints that return paginated results. It automatically
     * follows pagination links and combines all pages of results into a single array.
     * It's particularly useful for APIs that limit the number of records returned in a
     * single request.
     *
     * The method supports OData-style pagination by default (using '@odata.nextLink'),
     * but can be configured for other pagination styles by changing the $nextKey parameter.
     *
     * @param string $url Base API endpoint
     * @param array $params Optional query parameters to include in the request
     * @param string $nextKey JSON key that contains the URL for the next page (default: '@odata.nextLink')
     * @return array All combined response data from all pages
     */
    protected function getAllPages(string $url, array $params = [], string $nextKey = '@odata.nextLink'): array
    {
        $results = [];

        try {
            $fullUrl = $url . '?' . http_build_query($params);
            //echo "Full URL: " . $fullUrl . PHP_EOL;
            while ($fullUrl) {
                $response = $this->client->get($fullUrl);
                $data = json_decode($response->getBody(), true);

                // Append results from the current page
                if (isset($data['value']) && is_array($data['value'])) {
                    $results = array_merge($results, $data['value']);
                }

                // Determine next URL (if pagination is used)
                $fullUrl = $data[$nextKey] ?? null;
            }
        } catch (RequestException $e) {
            echo "Request failed: " . $e->getMessage();
        }

        return $results;
    }

    /**
     * Send a generic HTTP request to the API
     *
     * This is the core method that handles all API requests. It supports:
     * - Regular HTTP methods (GET, POST, PUT, PATCH, DELETE)
     * - Special 'CREATE' method that maps to POST for creating new resources
     * - Dry run mode that prevents destructive operations
     * - Error handling and logging
     *
     * The method automatically removes 'id' from the JSON payload to prevent issues
     * with some APIs that don't accept an ID in the request body.
     *
     * @param string $method HTTP method (GET, POST, PUT, PATCH, DELETE, CREATE)
     * @param string $url Endpoint URL
     * @param array $options Guzzle request options (headers, json, query, etc)
     * @return ResponseInterface|null The HTTP response or null if the request failed
     * @throws GuzzleException When there's an issue with the HTTP request
     */
    public function request(string $method, string $url, array $options = []): ?ResponseInterface
    {
        unset($options['json']['id']);
        if ((in_array($method, $this->destructiveMethods)) && $this->dry !== false) {
            // OK, we are set to Dry Run, so we dont allow any destructive operations
            $this->logger->info->apiRequest('Request to write data :' . $method, ['url' => $url, 'options' => $options]);
            $this->statusCode = 200;
        } else {
            if ($method === 'CREATE') {
                /*
                 * We use 'CREATE' as a method name to allow us to separate
                 * types of POST request
                 */
                $method = 'POST';
            }
            try {
                $this->return = $this->client->request($method, $url, $options);
            } catch (RequestException $e) {
                //echo "Request error: " . $e->getMessage();
                $this->logger->error->apiRequest("Request error : ", [$e->getResponse()->getBody()->getContents()]);
                $this->logger->error->apiRequest("Request Type : $method");
                $this->logger->error->apiRequest("Request URL : $url");
                $this->logger->error->apiRequest("Request Options : ", $options);
                $this->return = null;
            }
            $this->statusCode = $this->return ? $this->return->getStatusCode() : 0;
        }
        return $this->return;
    }

    /**
     * Set the list of HTTP methods considered destructive
     * 
     * Destructive methods are those that modify data and are prevented
     * from executing when dry run mode is enabled.
     *
     * @param array $destructiveMethods Array of HTTP method names
     * @return ApiClient This instance for method chaining
     */
    public function setDestructiveMethods(array $destructiveMethods): ApiClient
    {
        $this->destructiveMethods = $destructiveMethods;
        return $this;
    }

    /**
     * Set the dry run mode
     * 
     * When dry run mode is enabled (true), destructive operations
     * will be logged but not actually executed.
     *
     * @param bool $dry True to enable dry run mode, false to disable
     * @return ApiClient This instance for method chaining
     */
    public function setDry(bool $dry): ApiClient
    {
        $this->dry = $dry;
        return $this;
    }

    /**
     * Set the authentication URL for OAuth2 token endpoint
     *
     * @param string $authURL The full URL to the OAuth2 token endpoint
     */
    public static function setAuthURL(string $authURL): void
    {
        self::$authURL = $authURL;
    }

    /**
     * Set the client ID for OAuth2 authentication
     *
     * @param string $clientID The client ID provided by the API service
     */
    public static function setClientID(string $clientID): void
    {
        self::$clientID = $clientID;
    }

    /**
     * Set the client secret for OAuth2 authentication
     *
     * @param string $clientSecret The client secret provided by the API service
     */
    public static function setClientSecret(string $clientSecret): void
    {
        self::$clientSecret = $clientSecret;
    }

    /**
     * Set the grant type for OAuth2 authentication
     *
     * @param string $grantType The OAuth2 grant type (e.g., 'client_credentials', 'password', etc.)
     */
    public static function setGrantType(string $grantType): void
    {
        self::$grantType = $grantType;
    }

    /**
     * Set the scope for OAuth2 authentication
     *
     * @param string $scope The OAuth2 scope defining the access permissions
     */
    public static function setScope(string $scope): void
    {
        self::$scope = $scope;
    }

    /**
     * Get the list of HTTP methods considered destructive
     *
     * @return array Array of HTTP method names
     */
    public function getDestructiveMethods(): array
    {
        return $this->destructiveMethods;
    }

    /**
     * Check if dry run mode is enabled
     *
     * @return bool True if dry run mode is enabled, false otherwise
     */
    public function isDry(): bool
    {
        return $this->dry;
    }

    /**
     * Get all API client instances
     *
     * @return array Array of API client instances indexed by endpoint
     */
    public static function getInstances(): array
    {
        return self::$instances;
    }

    /**
     * Get the response from the last API request
     *
     * @return ResponseInterface|null The HTTP response or null if the last request failed
     */
    public function getReturn(): ?ResponseInterface
    {
        return $this->return;
    }

    /**
     * Get the HTTP status code from the last API request
     *
     * @return int The HTTP status code or 0 if the last request failed
     */
    public function getStatusCode(): int
    {
        return $this->statusCode;
    }



}
